//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file TransmissionMedium.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::TransmissionMedium

	\sa TransmissionMedium.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/


#ifndef ODEMX_TRANSMISSION_INCLUDED
#define ODEMX_TRANSMISSION_INCLUDED

#include <odemx/protocol/NetTopology.h>


namespace odemx {

	class ProtocolMessage;
	class ProtocolSimulation;
	class BroadcastTransmission;
	
	class TransmissionMediumObserver;

	
	/** \class TransmissionMedium

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %TransmissionMedium manages the NetTopology and message transmissions 
		
		\sa ProtocolMessage, TransmissionMedium

		%TransmissionMedium is at the center of a ProtocolSimulation. All messages 
		pass through the medium and will be forwarded to their next hop at 
		the given message transmission time.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class TransmissionMedium: public NetTopology,
							  public DefLabeledObject,
							  public Observable<TransmissionMediumObserver>,
							  public virtual TraceProducer { 

	public:
		
		/**
			\brief Construction
			\param sim
				pointer to the Simulation object
			\param l
				label of this object
		*/
		TransmissionMedium( ProtocolSimulation* sim, Label l, TransmissionMediumObserver* o = 0 );
		
		/// Destruction
		virtual ~TransmissionMedium();


		/** \name Protocol message transmission
		
			These functions provide the means to model message transmissions
			between NetNodes. Transmission duration is modeled by delayed 
			scheduling of transmission events. Transmission failure is reported.
			
			@{
		*/
		/**
			\brief transmit a ProtocolMessage
			\param m
				message to be transmitted over the medium
			\return
				\c false if message was lost during transmission

			This function models transmission of messages with some 
			probability of message loss, and should then return false.
			The transmission time of a message is computed and transmission
			events are scheduled depending on the message type. For
			unicast messages, connectivity is checked here.
		*/
		virtual bool transmit( ProtocolMessage* m );

		/**
			\brief check connection between nodes
			\param sendNode
				current message sender
			\param receiveNode
				next message receiver
			\return
				\c false if the nodes are not connected

			This function determines whether there is a connection
			set between two nodes, regardless of connection quality.
		*/
		virtual bool noConnectionSet( NetNode* sendNode, NetNode* receiveNode );

		/**
			\brief check current connectivity between nodes
			\param sendNode
				current message sender
			\param receiveNode
				next message receiver
			\return
				\c false if message would be lost

			This function determines if there currently is a connection
			between the two nodes, based on connection quality. This is
			modeled by requesting a sample from a random variable.
			If sample() returns 0, the message is lost.
		*/
		virtual bool badConnectivity( NetNode* sendNode, NetNode* receiveNode );		
		
		
		/**
			\brief schedule a unicast transmission event
			\param m
				message to be transmitted over the medium

			This function models transmission of messages with a
			delay specified by the computed transmission time of
			the given message.
		*/
		virtual void scheduleUnicastTransmission( ProtocolMessage* m );

		
		/**
			\brief schedule a broadcast transmission event
			\param m
				message to be transmitted over the medium

			This function models transmission of broadcast messages with a
			delay specified by the computed transmission time of
			the given message.
		*/
		virtual void scheduleBroadcastTransmission( ProtocolMessage* m );
		//@}
		
		/// get the trace object
		Trace* getTrace() const;
		/// get the ProtocolSimulation object
		ProtocolSimulation* getSimulation() const;
		/// get the current simulation time
		SimTime getCurrentTime() const;
		
		/// increase the collision counter by \p c
		void updateCollisionCount( unsigned int c = 1 );
		
	protected:
		
		friend class UnicastTransmission;
		friend class BroadcastTransmission;
		
		/// report unicast message loss, update statistics
		void unicastFailure( ProtocolMessage* m, NetNode* sender, NetNode* receiver );
		/// report broadcast message loss, update statistics
		void broadcastFailure( ProtocolMessage* m, NetNode* sender, NetNode* receiver );

	private:

		/// pointer to the ProtocolSimulation object
		ProtocolSimulation* env;
		
		friend void ProtocolSimulation::report( Report* r );
		
		int messageCount;
		int broadcastCount;
		int unicastCount;
		int successfulBroadcastCount;
		int successfulUnicastCount;
		int failedBroadcastCount;
		int failedUnicastCount;
		int collisionCount;
		
	public:
		
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetTopology calls.
			A TraceConsumer can use these constants to identify  
			trace events sent by NetTopology.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;
		static const MarkType markTransmit;
		static const MarkType markScheduleUnicast;
		static const MarkType markScheduleBroadcast;
		
		static const TagId baseTagId;

		static const Tag tagFrom;
		static const Tag tagTo;
		static const Tag tagAt;

		//@}

	};

	
	
	/** \interface TransmissionMediumObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for TransmissionMedium specific simulation events.

		<!-- [\note {notes}]* -->

		\sa NetTopology, TransmissionMedium

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class TransmissionMediumObserver {
		
	public:
		/// Destructor
		virtual ~TransmissionMediumObserver() {}
		
		/// Construction
		virtual void onCreate( TransmissionMedium* sender ) {}
		/// Destruction
		virtual void onDestroy( TransmissionMedium* sender ) {}
		/// Transmit message
		virtual void onTransmit( TransmissionMedium* sender, ProtocolMessage* m ) {}
		/// Schedule a unicast event
		virtual void onScheduleUnicast( TransmissionMedium* sender, ProtocolMessage* m, SimTime t ) {}
		/// Schedule a broadcast event
		virtual void onScheduleBroadcast( TransmissionMedium* sender, ProtocolMessage* m, SimTime t ) {}	
	};

	
}


#endif /* ODEMX_TRANSMISSION_INCLUDED */
