//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProtocolLayer.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::ProtocolLayer, odemx::ProtocolLayerObserver, odemx::ProtocolLayerManager 

	\sa ProtocolLayer.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/




#ifndef ODEMX_PROTOCOL_LAYER_INCLUDED
#define ODEMX_PROTOCOL_LAYER_INCLUDED


#include <odemx/util/LabeledObject.h>
#include <odemx/util/Observable.h>
#include <odemx/util/Trace.h>

#include <list>
#include <vector>
#include <string>

namespace odemx {
	
	// forward declarations
	class ProtocolEntity;
	class ProtocolLayer;
	class ProtocolLayerObserver;
	class ProtocolLayerManager;
	class ProtocolSimulation;
	class NetNode;
	
	/** \var typedef std::list< ProtocolEntity* > ProtocolEntityList;
		
		\brief list containing protocol entities
	*/
	typedef std::list< ProtocolEntity* > ProtocolEntityList;

	/** \var typedef std::vector< ProtocolLayer* > ProtocolLayerList;
		
		\brief vector containing protocol layers
	*/
	typedef std::vector< ProtocolLayer* > ProtocolLayerList;
	
	

	/** \class ProtocolLayer

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %ProtocolLayer is the base class for layers in protocol simulations
		
		\sa ProtocolLayerManager, ProtocolLayerT, ProtocolEntity, ProtocolSimulation

		ProtocolLayers are a means of structuring the protocol stack in
		a simulation model. All of the simulation model's protocol layers 
		are to be derived from this class. Each layer has different tasks 
		to fulfil, which is achieved by employing ProtocolEntities as active 
		components. The layers of a simulation model are managed by a 
		ProtocolLayerManager. 

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/

	//  
	class ProtocolLayer: public virtual TraceProducer,
						 public DefLabeledObject,
						 public Observable<ProtocolLayerObserver> {
	public:
	
		
		/**
			\brief Construction
			\param sim
				pointer to the ProtocolSimulation object
			\param l
				Label of the layer
			\param o
				initial observer of this object
		*/
		ProtocolLayer( ProtocolSimulation* sim, Label l, ProtocolLayerObserver* o = 0 );
		
		/// Destruction
		virtual ~ProtocolLayer();
	
		
		/**
			\brief creation of a new entity for this layer
			\param owner
				pointer to the owning NetNode
				
			This function must be defined by derived layer classes.
			This declaration ensures that the function can be called
			polymorphically on a ProtocolEntity pointer.
		*/
		virtual ProtocolEntity* createNewEntity( NetNode* owner ) = 0;
		
		
		/// get a reference to the list of entities belonging to a layer
		ProtocolEntityList& getEntityList();
		
		/// add a ProtocolEntity to the ProtocolLayer
		void addEntity( ProtocolEntity* e );
		
		/// remove a ProtocolEntity from the layer 
		void removeEntity( ProtocolEntity* e );
	
		/// print a list of the layer's entities to stdout 
		void printEntityList();
		
		/// get a pointer to the ProtocolSimulation object
		ProtocolSimulation* getSimulation() const;
		
		/// get a pointer to the Trace object
		Trace* getTrace() const;
		
	private:
		/// pointer to the ProtocolSimulation object
		ProtocolSimulation* env;
		/// list containing all of the layer's entities
		ProtocolEntityList entityList;
		
		
	public:
		
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetTopology calls.
			A TraceConsumer can use these constants to identify  
			trace events sent by NetTopology.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;
		static const MarkType markAddEntity;
		static const MarkType markRemoveEntity;
		//@}

	};


	
	/** \interface ProtocolLayerObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for ProtocolLayer specific simulation events.

		<!-- [\note {notes}]* -->

		\sa ProtocolLayer

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class ProtocolLayerObserver {
		
	public:
		/// Destructor
		virtual ~ProtocolLayerObserver() {}
		
		/// Construction
		virtual void onCreate( ProtocolLayer* sender ) {}
		/// Destruction
		virtual void onDestroy( ProtocolLayer* sender ) {}
		/// Register new NetNode
		virtual void onAddEntity( ProtocolLayer* sender, ProtocolEntity* e ) {}
		/// Remove an existing NetNode
		virtual void onRemoveEntity( ProtocolLayer* sender, ProtocolEntity* e ) {}
	};

	


	/** \class ProtocolLayerManager

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %ProtocolLayerManager handles the layers in protocol simulations
		
		\sa ProtocolLayerManager, ProtocolLayer, ProtocolSimulation

		The internal class ProtocolLayerManager stores a list of 
		ProtocolLayers and provides access to layers if needed. It is
		mainly used in the automatic creation of a NetNode's entities and
		in several internal functions that require access to specific layers.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class ProtocolLayerManager {
	private:
	
		friend class ProtocolSimulation;
		friend class ProtocolLayer;
		friend class NetNode;
		
		
		/// add a new protocol layer to the internal list
		void addLayer( ProtocolLayer* layer );
		
		/**
			\brief get pointer to a layer requested by its name
			\param l
				label of the requested layer
			\return
				pointer to the requested layer
		*/
		ProtocolLayer* getLayerByLabel( Label l );
		
		/// get a reference to the internal ProtocolLayerList 
		ProtocolLayerList& getLayerList();
		
	private:
		
		/// associative array of names and layer pointers
		std::map< std::string, ProtocolLayer* > layerNameMap;
		/// list of layers in top-down order
		ProtocolLayerList allLayersTopDown;
	};
	
	
	
	/** \class ProtocolLayerT

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %ProtocolLayerT is a template that can handle user-defined entity types
		
		\sa ProtocolLayer, ProtocolSimulation

		The internal class ProtocolLayerT implements layers with user-defined
		entity types. The template argument specifies the type of ProtocolEntity
		to be created when \c createNewEntity() is called.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	template < typename EntityType >
	class ProtocolLayerT: public ProtocolLayer {
	public:
		
		/**
			\brief Construction
			\param sim
				pointer to the ProtocolSimulation object
			\param l
				Label of the layer
			\param o
				initial observer of this object
		*/
		ProtocolLayerT( ProtocolSimulation* sim, Label l, ProtocolLayerObserver* o = 0 ):
			ProtocolLayer( sim, l, o )
			{}
		
		
		/**
			\brief creates a new entity of the given type
			\param owner
				pointer to the owning NetNode
			\return
				pointer to newly created ProtocolEntity

			This function dynamically allocates a new object of the given
			EntityType, adds it to the list of entities belonging to this
			layer, and returns a pointer to the object.
		*/
		virtual EntityType* createNewEntity( NetNode* owner ) {
			
			EntityType* newEntity = new EntityType( getSimulation(), this, owner );
			
			this -> addEntity( newEntity );
			
			return newEntity;
		}
	};
	

}




#endif /* ODEMX_PROTOCOL_LAYER_INCLUDED */
