//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Trace.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/10

	\brief Trace mechanism declaration

	\sa Trace.cpp

	A Trace mechanism is used to report events and state changes of objects in a generalised
	way. Objects (TraceProducer) generate a sequence of marks, which are send to a
	trace manager (Trace). The trace manager broadcasts the marks to all registered
	trace handler (TraceConsumer).
	A Mark is defined by its type (MarkType). A sender can use
	one of the predefined mark-functions to send a mark or compose a mark out of tags.

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_TRACE_INCLUDED
#define ODEMX_TRACE_INCLUDED

#include <odemx/util/LabeledObject.h>
#include <odemx/util/TypedObject.h>

#include <list>
#include <cassert>

namespace odemx {
	class Trace;	

	/// Mark type information
	typedef unsigned long MarkTypeId;

	/// Tag identification
	typedef unsigned long TagId;

	/** \class MarkType

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief MarkType describes a trace mark.

		<!-- [\note {notes}]* -->

		\sa Trace TraceProducer TraceConsumer Tag

		A mark is defined by its MarkType.
		The MarkType can be a string (name) and/or a number.
		A MarkType is always associated to a definition scope.

		\par Usage of MarkType:
		\code
void sendMark(TraceProducerX* s, Mark m) {}

int main ()
{
	TraceProducerX prod;
	static const MarkTypeId FingerPrint = MarkType::NOID + 100;
	sendMark(&prod, MarkType("Footprint", typeid(X)) );
	sendMark(&prod, MarkType(FingerPrint, typeid(X));
	sendMark(&prod, MarkType("FootPrint", FingerPrint, typeid(X)));
}
		\endcode

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class MarkType {
		const char* name; ///< MarkType name
		MarkTypeId id; ///< MarkType id
		const std::type_info& scope; ///< MarkType definition scope
	public:
		/// \name Construction
		//@{
		/**
			\brief constructor

			\param n
				name of MarkType
			\param s
				scope of MarkType
		*/
		MarkType(const char* n, const std::type_info& s) : name(n), id(NOID), scope(s) {assert(name!=NONAME);}

		/**
			\brief constructor

			\param i
				id of MarkType
			\param s
				scope of MarkType
		*/
		MarkType(MarkTypeId i, const std::type_info& s) : name(NONAME), id(i), scope(s) {assert(id!=NOID);}

		/**
			\brief constructor

			\param n
				name of MarkType
			\param i
				id of MarkType
			\param s
				scope of MarkType
		*/
		MarkType(const char* n, MarkTypeId i, const std::type_info& s) : name(n), id(i), scope(s) {assert(name!=NONAME || id!=NOID);}
		//@}

		/// get MarkType name
		const char* getName() const {return name;}

		/// get MarkType id
		MarkTypeId getId() const {return id;}

		/// get MarkType definition scope
		const std::type_info& getScope() const {return scope;}

		//
		// special handles
		//
		static const MarkTypeId NOID;
		static const char* NONAME;
	};

	/** \class Tag

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief A Tag is used to build composed marks.

		<!-- [\note {notes}]* -->

		\sa Trace TraceProducer TraceConsumer MarkType

		\par Usage of Tag:
		\code
void addTag(Tag t) {}

int main() 
{
	static const TagId tagNumber = Tag::NOID + 100;
	addTag("blabla");
	addTag(tagNumber);
}
		\endcode

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Tag {
		/// Tag name
		const char* name;

		/// Tag id
		TagId id;
	public:
		/**
			\name Construction
		
			The constructors are used for automatic typecasts.

			@{
		*/	
		/**
			\brief constructor

			\param n
				Tag name
		*/
		Tag(const char* n) : name(n), id(NOID) {assert(name!=NONAME);}

		/**
			\brief constructor

			\param i
				Tag id
		*/
		Tag(TagId i) : name(NONAME), id(i) {assert(id!=NOID);}
		//@}

		friend bool operator ==(const Tag& t1, const Tag& t2);

		/// get Tag name
		const char* getName() const {return name;}

		/// get Tag id
		TagId getId() const {return id;}

		//
		// special handle
		//
		static const TagId NOID;
		static const char* NONAME;
	};

	/** \interface TraceProducer

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief A TraceProducer generates marks that form the trace.

		<!-- [\note {notes}]* -->

		\sa Trace TraceConsumer MarkType Tag

		\copydoc Trace.h
		
		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class TraceProducer : public virtual LabeledObject, public virtual TypedObject {
	public:
		/**
			\brief A TraceProducer must have access to a Trace object.

			\return pointer to a Trace object
		*/
		virtual Trace* getTrace() const = 0;
	};

	/** \interface TraceConsumer

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief A TraceConsumer receives marks from the Trace it is added to.

		<!-- [\note {notes}]* -->

		\sa Trace TraceProducer MarkType Tag

  		\copydoc Trace.h

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class TraceConsumer {
	public:
		virtual ~TraceConsumer() { }
	private:
		// Only Trace may call any of these methods.
		friend class Trace;

		/**
			\name Simple trace marks.
		
			Simple trace marks should be handled by the following methods.

			@{
		*/		
		/**
			\brief handle most simple trace mark

			\param sender
				pointer to sender of this mark
			\param m
				type of this mark
			\param comment
				comments
		*/
		virtual void mark(const TraceProducer* sender, const MarkType* m, const char* comment = 0) = 0;

		// Marks that include information about associations of the producer:
		/// handle marks that includes information about associations of the producer
		virtual void mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* partner, const char* comment = 0) = 0;
		/// handle mark that includes information about associations of the producer
		virtual void mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment = 0) = 0;

		// Marks that include information about attributes of the producer:
		/// handle bool-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment = 0) = 0;
		/// handle char-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, char newValue, char oldValue, const char* comment = 0) = 0;
		/// handle short-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, short newValue, short oldValue, const char* comment = 0) = 0;
		/// handle int-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, int newValue, int oldValue, const char* comment = 0) = 0;
		/// handle long-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, long newValue, long oldValue, const char* comment = 0) = 0;
		/// handle float-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, float newValue, float oldValue, const char* comment = 0) = 0;
		/// handle double-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, double newValue, double oldValue, const char* comment = 0) = 0;
		/// handle const char*-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment = 0) = 0;
		
		/// handle unsigned char-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned char newValue, unsigned char oldValue, const char* comment = 0) = 0;
		/// handle unsigned short-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned short newValue, unsigned short oldValue, const char* comment = 0) = 0;
		/// handle unsigned int-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned int newValue, unsigned int oldValue, const char* comment = 0) = 0;
		/// handle unsigned long-attribute changes
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned long newValue, unsigned long oldValue, const char* comment = 0) = 0;
		//@}

		/**
			\name Composed trace marks.

			Composed trace marks ar transmitted with the following methods.
			A composed trace mark can be interpreted as a tree of tag-value pairs.

			@{
		*/
		/// indicates the construction of a new mark
		virtual void beginMark(const TraceProducer* sender, const MarkType* m) = 0;
		/// signals the end of construction
		virtual void endMark(const char* comment = 0) = 0;

		/**
			\brief called to begin of the composed Tag \p t.

			\note A TraceConsumer may rely on correct order of beginTag()/endTag() calls (XML).
		*/
		virtual void beginTag(Tag t) = 0;
		/// end of Tag \p t
		virtual void endTag(Tag t) = 0;
		
		virtual void addTag(Tag t, bool value) = 0; ///< handle a Tag-bool value pair
		virtual void addTag(Tag t, char value) = 0; ///< handle a Tag-char value pair
		virtual void addTag(Tag t, short value) = 0; ///< handle a Tag-short value pair
		virtual void addTag(Tag t, int value) = 0; ///< handle a Tag-int value pair
		virtual void addTag(Tag t, long value) = 0; ///< handle a Tag-long value pair
		virtual void addTag(Tag t, float value) = 0; ///< handle a Tag-float value pair
		virtual void addTag(Tag t, double value) = 0; ///< handle a Tag-double value pair
		virtual void addTag(Tag t, const char* value) = 0; ///< handle a Tag-const char* value pair
		virtual void addTag(Tag t, const TraceProducer* value) = 0; ///< handle a Tag-TraceProducer* value pair

		virtual void addTag(Tag t, unsigned char value) = 0; ///< handle a Tag-unsigned char value pair
		virtual void addTag(Tag t, unsigned short value) = 0; ///< handle a Tag-unsigned short value pair
		virtual void addTag(Tag t, unsigned int value) = 0; ///< handle a Tag-unsigned int value pair
		virtual void addTag(Tag t, unsigned long value) = 0; ///< handle a Tag-unsigned long value pair
		//@}
		
		/// \name Trace control
		//@{
		/** 
			\brief This call marks the beginning of a trace
			
			A TraceConsumer should initialise for a new trace.
		*/
		virtual void startTrace() = 0;

		/**
			\brief This call is send after the traced finished

			A TraceConsumer may clean up.
		*/
		virtual void stopTrace() = 0; 

		/**
			\brief This call signals a break in trace
			
			A TraceConsumer should not produce output after this call is send,
			but await further marks for updating internal data if required.
			(An open composed mark will be closed before this call.)
		*/
		virtual void pauseTrace() = 0;

		/**
			\brief This call signals the end of a break in trace
			
			A TraceConsumer should update its output if required,
			and report following marks.
		*/
		virtual void continueTrace() = 0;
		//@}
	};

	/** \class Trace

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief The Trace receives marks from TraceProducer objects and informs registered TraceConsumer about these marks.

		<!-- [\note {notes}]* -->

		\sa TraceConsumer TraceProducer MarkType Tag

		\copydoc Trace.h

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Trace {
	public:
		Trace(); ///< Construction
		~Trace(); ///< Destruction

		/**
			\name Simple trace marks

			Simple trace marks are reported by the following methods.

			@{
		*/
		/**
			\brief send most simple trace mark

			\param sender
				pointer to sender of this mark
			\param m
				type of this mark
			\param comment
				comments
		*/
		void mark(const TraceProducer* sender, MarkType m, const char* comment = 0);

		// Marks that include information about associations of the producer:
		/// send marks that includes information about associations of the producer 
		void mark(const TraceProducer* sender, MarkType m, const TraceProducer* partner, const char* comment = 0);
		/// send marks that includes information about associations of the producer
		void mark(const TraceProducer* sender, MarkType m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment = 0);

		// Marks that include information about attributes of the producer:
		/// send bool-attribute changes
		void mark(const TraceProducer* sender, MarkType m, bool newValue, bool oldValue, const char* comment = 0);
		/// send char-attribute changes
		void mark(const TraceProducer* sender, MarkType m, char newValue, char oldValue, const char* comment = 0);
		/// send short-attribute changes
		void mark(const TraceProducer* sender, MarkType m, short newValue, short oldValue, const char* comment = 0);
		/// send int-attribute changes
		void mark(const TraceProducer* sender, MarkType m, int newValue, int oldValue, const char* comment = 0);
		/// send long-attribute changes
		void mark(const TraceProducer* sender, MarkType m, long newValue, long oldValue, const char* comment = 0);
		/// send float-attribute changes
		void mark(const TraceProducer* sender, MarkType m, float newValue, float oldValue, const char* comment = 0);
		/// send double-attribute changes
		void mark(const TraceProducer* sender, MarkType m, double newValue, double oldValue, const char* comment = 0);
		/// send const char*-attribute changes
		void mark(const TraceProducer* sender, MarkType m, const char* newValue, const char* oldValue, const char* comment = 0);
		
		/// send unsigned char-attribute changes
		void mark(const TraceProducer* sender, MarkType m, unsigned char newValue, unsigned char oldValue, const char* comment = 0);
		/// send unsigned short-attribute changes
		void mark(const TraceProducer* sender, MarkType m, unsigned short newValue, unsigned short oldValue, const char* comment = 0);
		/// send unsigned int-attribute changes
		void mark(const TraceProducer* sender, MarkType m, unsigned int newValue, unsigned int oldValue, const char* comment = 0);
		/// send unsigned long-attribute changes
		void mark(const TraceProducer* sender, MarkType m, unsigned long newValue, unsigned long oldValue, const char* comment = 0);
		//@}
		
		/**	\name Composed trace marks
			Composed trace marks are build with the following methods.
			A composed trace mark can be interpreted as an sequence of tag-value pairs.

			A structure can be build with beginTag() and endTag().
			It is recommended but not surveyed that for every beginTag(X) an endTag(X) is called in 
			the appropriate order (XML).	A TraceConsumer may rely on a correct order of 
			beginTag() - endTag() calls.

			@{
		*/
		/**
			\brief begin a new composed mark

			The construction of a new composed mark starts with beginMark(...). 
			If a older mark is still in construction [endMark(...) was not called]
			it is closed.
		*/
		void beginMark(const TraceProducer* sender, MarkType m);

		/**
			\brief end of composed mark

			The construction of a composed mark is finished by endMark().
		*/
		void endMark(const char* comment = 0);

		// With the following functions the composed mark is constructed.
		/**
			\brief called to begin of the composed Tag \p t.

			\note A TraceConsumer may rely on correct order of beginTag()/endTag() calls (XML).
		*/
		void beginTag(Tag t); ///< begin a tag
		/// end of Tag \p t
		void endTag(Tag t); ///< end tag

		void addTag(Tag t, bool value); ///< add a tag with a boolean value
		void addTag(Tag t, char value); ///< add a tag with a char value
		void addTag(Tag t, short value); ///< add a tag with a short value
		void addTag(Tag t, int value); ///< add a tag with an int value
		void addTag(Tag t, long value); ///< add a tag with a long value
		void addTag(Tag t, float value); ///< add a tag with a float value
		void addTag(Tag t, double value); ///< add a tag with a double value
		void addTag(Tag t, const char* value); ///< add a tag with a const char* value
		void addTag(Tag t, const TraceProducer* value); ///< add a tag with a TraceProducer* value

		void addTag(Tag t, unsigned char value); ///< add a tag with an unsigned char value
		void addTag(Tag t, unsigned short value); ///< add a tag with an unsigned short value
		void addTag(Tag t, unsigned int value); ///< add a tag with an unsigned int value
		void addTag(Tag t, unsigned long value); ///< add a tag with an unsigned long value
		//@}

		/// \name TraceConsumer management
		//@{		
		void addConsumer(TraceConsumer* c); ///< add Consumer		
		void removeConsumer(TraceConsumer* c); ///< remove Consumer
		//@}

		/**
			\name Trace control

			The trace is started with startTrace() end finished with stopTrace() or destruction.
			pauseTrace() and continueTrace() should be used to signal temporary deactivation of
			trace generation. 
			
			\note An open composed mark will be closed by pauseTrace and stopTrace.

			@{
		*/
		void startTrace(); ///< start trace
		void stopTrace(); ///< stop trace
		void pauseTrace(); ///< break trace
		void continueTrace(); ///< continue trace
		//@}

	private:
		std::list<TraceConsumer*> cons; ///< TraceConsumer
		enum STATE{CREATED, RUNNING, PAUSED, FINISHED} state; ///< Trace state
		bool openMark; ///< currently open composed mark

		/// \name help procedures
		//@{
		/// send a simple mark to all consumer
		template <class T>
		void broadcastMark(const TraceProducer* sender, MarkType m, T newValue, T oldValue, const char* comment);

		/// send a tag to all consumer
		template <class T>
		void boradcastTagValue(Tag t, T value);

		/// handle trace pause
		bool sendMarks() {return (state==RUNNING || state==PAUSED);}
		//@}
	};
}

#endif

