//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Waitq.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/26

	\brief Definition of odemx::Waitq

	\sa Waitq.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_WAITQ_INCLUDED
#define ODEMX_WAITQ_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/Observable.h>
#include <odemx/util/Report.h>
#include <odemx/util/StatisticObject.h>
#include <odemx/statistics/Statistics.h>
#include <odemx/synchronization/Queue.h>

namespace odemx  {
	class WaitqObserver;

	/** \class Waitq

		\ingroup synch

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Waitq realises a master slave synchronisation,
		where the master gets control after successful synchronisation.

		\note Waitq supports Observation
		\note Waitq supports Trace
		\note Waitq supports Report

		<!-- [\sa {references to other classes}]* -->

		Waitq realises a master slave synchronisation scheme. Two processes
		are synchronised with each other. One plays the role of a master process
		and gets control (execution) after successful synchronisation. The other
		(slave) is frozen and returned to the master process. A master process
		can provide a selection function to synchronise with particular processes
		only. If multiple processes could be used for synchronisation process-
		priority and FIFO-strategy are used to choose the process.
		Waitq is used for general process to process synchronisation.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Waitq : public Observable<WaitqObserver>,
				  public DefLabeledObject,
				  public StatisticObject,
				  public virtual TraceProducer,
				  public virtual ReportProducer
	{
	public:
		/**
			\brief Construction
			\param s
				pointer to Simulation object
			\param l
				label of this object
			\param o
				initial observer
		*/
		Waitq(Simulation* s, Label l, WaitqObserver* o = 0);
		/// Destruction
		~Waitq();

		/**
			\name Master-slave synchronisation
			@{
		*/
		/**
			\brief Wait for activation by a 'master' process

			A process calling wait() is deactivated and passed
			over to a process synchronising with coopt() or avail(). If
			a slave process is interrupted before a successful synchronisation
			it is reactivated and the function returns false. (An interrupted
			slave process does not change user and waiting time statistics, but has
			influence on queue statistics.)
		*/
		bool wait();

		/**
			\brief Get a 'slave' process

			A master process uses coopt() to synchronise with a slave process.
			The master can provide a 'selection function' to synchronise with
			a specific slave process. Until there is a suitable slave process
			available the master process is blocked. If a blocked master process
			is interrupted the synchronisation attempt is cancelled and coopt()
			returns 0. (An interrupted master process does not change user and wait
			time statistics, but has influence on queue statistics.)

			\warning A master may not call interrupt() to reactivate a received slave().
		*/
		Process* coopt(Selection sel=0);

		/**
			\brief Get available slaves without blocking (optional: select slave)

			A master process can use avail() to get a suitable slave process if
			available. Otherwise avail() returns 0.

			\warning A master may not call interrupt() to reactivate a received slave().
		*/
		Process* avail(Selection sel=0);
		//@}

		/// List of blocked slaves
		const std::list<Process*>& getWaitingSlaves() const {return slaveWait.getList();}
		/// List of blocked masters
		const std::list<Process*>& getWaitingMasters() const {return masterWait.getList();}

		/**
			\name Statistics

			Statistics

			@{
		*/
		/// Reset statistics
		virtual void reset();

		/// Number of synchronisation
		unsigned int getNumberOfSynch() const {return synchs;}

		/// Without wait served slaves
		unsigned int getZeroWaitSlaves() const {return zeroSlaves;}

		/// Without wait served masters
		unsigned int getZeroWaitMasters() const  {return zeroMasters;}

		/// Average slave waiting time
		double getAVSlaveWaitTime() const {return sumSlaveWait / synchs;}

		/// Average master waiting time
		double getAVMasterWaitTime() const {return sumMasterWait / synchs;}

		//@}


		/// Pointer to trace
		virtual Trace* getTrace() const {return env;}

		/**
			\name Waitq specific marks
			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markWait;
		static const MarkType markCooptFail;
		static const MarkType markCooptSucceed;
		static const MarkType markAvailFail;
		static const MarkType markAvailSucceed;

		static const MarkType markCooptSelFail;
		static const MarkType markCooptSelSucceed;
		static const MarkType markAvailSelFail;
		static const MarkType markAvailSelSucceed;

		static const TagId baseTagId;

		static const Tag tagMaster;
		static const Tag tagSlave;
		//@}

		/// Generate report
		virtual void report(Report* r);

		// Implementation
	private:
		/// simulation
		Simulation* env;
		/// master management
		Queue masterWait;
		/// slave management
		Queue slaveWait;

		// Statistics
		unsigned int zeroMasters; /// without wait served master
		unsigned int zeroSlaves; /// without wait served slaves

		unsigned int synchs; /// number of synchronisation

		double sumMasterWait; /// average master waiting time
		double sumSlaveWait; /// average slave waiting time

		// help methods
		/// current process
		Process* getCurrentProcess() {return env->getCurrentProcess();}
		/// find slave
		Process* getSlave(Process* master=0, Selection sel=0);
		/// trace marks
		void traceSUCCEED(MarkType m, Process* master, Process* slave);
		/// statistics
		void updateStatistics(Process* master, Process* slave);

		/// master synch. implementation
		Process* synch(bool wait, Selection sel);
	};

	/** \interface WaitqObserver

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for Waitq

		<!-- [\note {notes}]* -->

		\sa Waitq

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class WaitqObserver {
	public:
		virtual ~WaitqObserver() { }
		/// Construction
		virtual void onCreate(Waitq* sender) {}
		/// Destruction
		virtual void onDestroy(Waitq* sender) {}

		/// Process is synchronising as slave
		virtual void onWait(Waitq* sender, Process* slave) {}
		/// Process is blocked in synchronisation as master
		virtual void onCooptFail(Waitq* sender, Process* master) {}
		/// Process succeeds in synchronisation as master, getting slave
		virtual void onCooptSucceed(Waitq* sender, Process* master, Process* slave) {}
		/// master Process doesn't find any slave
		virtual void onAvailFail(Waitq* sender, Process* master) {}
		/// master Process find some slave
		virtual void onAvailSucceed(Waitq* sender, Process* master) {}

		/// Process is blocked in synchronisation as master for a particular slave
		virtual void onCooptSelFail(Waitq* sender, Process* master) {}
		/// Process succeeds in synchronisation as master, getting a particular slave
		virtual void onCooptSelSucceed(Waitq* sender, Process* master, Process* slave) {}
		/// master Process doesn't find a particular slave
		virtual void onAvailSelFail(Waitq* sender, Process* master) {}
		/// master Process find a particular slave
		virtual void onAvailSelSucceed(Waitq* sender, Process* master) {}
	};
}

#endif

