//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Rdist.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/09

	\brief Declaration of random continuous number generators

	\sa Rdist.cpp

	Continuous random number generator.

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_RDIST_INCLUDED
#define ODEMX_RDIST_INCLUDED

#include <odemx/random/Dist.h>

namespace odemx {
	/**
		\defgroup rdist Continuous distributions
		\ingroup random

		A group of random number generators which produce continuous (double)
		random numbers of different distributions.
	*/

	/** \interface Rdist

		\ingroup rdist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Interface for continuous distributions

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		%Rdist declares the 'double sample()' method for continuous
		distribution classes. All derived classes implement this
		method to provide the random numbers in different distributions.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Rdist: public Dist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object
		*/
		Rdist(DistContext* c, Label title) : Dist(c, title) {};
		/// Destruction
		virtual ~Rdist() {};

		/**
			\brief Get next random number

			This function returns the next random number.
			It is implemented in derived classes which generate
			random numbers in different distributions.
		*/
		virtual double sample()=0;
	};

	/** \class Normal

		\ingroup rdist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Normal distribution of random numbers

		\note Normal from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Normal provides a series of normal ('Gauss')
		distributed random numbers. The parameter mean
		and divergence are set in the constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Normal: public Rdist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object

			\param na
				a mean-value

			\param nb
				b divergence

			The parameter \c na and \c nb define the mean-value
			and the divergence of the normal distribution generated
			by %Normal.
		*/
		Normal(DistContext* c, Label title, double na, double nb);
		/// Destruction
		virtual ~Normal(){};

		/// Get next random number
		virtual double sample();

		/// Get parameter na
		double getA() {return a;}
		/// Get parameter nb
		double getB() {return b;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	private:
		double  zyqu, zyqv;
		int zyqeven;

	protected:
		double a, b;
		int uses;
	};

	/** \class Negexp

		\ingroup rdist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Negative exponential distribution of random numbers

		\note Negexp from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Negexp provides a series of negative exponential
		distributed random numbers.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Negexp: public Rdist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object

			\param na
				inverse expected value

		*/
		Negexp(DistContext* c, Label title, double na);
		/// Destruction
		virtual ~Negexp(){};

		/// Get next random number
		virtual double sample();

		/// Get parameter \c na
		double getA() {return a;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	protected:
		double  a;
		int uses;
	};


	/** \class Uniform

		\ingroup rdist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Uniform distribution of random numbers

		\note Uniform from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Uniform provides a series of uniform
		distributed random numbers in the interval [a, b).
		The parameter a and b are set in the constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Uniform: public Rdist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object

			\param ua
				lower bound a

			\param ub
				upper bound b

			The parameters \c ua and \c ub define the interval
			of the distribution.
		*/
		Uniform(DistContext* c, Label title, double ua, double ub);
		/// Destruction
		virtual ~Uniform(){};

		/// Get next random number
		virtual double sample();

		/// Get parameter a
		double getA() {return a;}
		/// Get parameter b
		double getB() {return b;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	private:
		double  zyqspan;

	protected:
		double a, b;
		int uses;
	};

	/** \class Erlang

		\ingroup rdist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Erlang distribution of random numbers

		\note Erlang from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Erlang provides a series of Erlang-distributed
		random numbers. The parameters are set in the
		constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Erlang: public Rdist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				Pointer to DistContext object

			\param ea

			\param eb
		*/
		Erlang(DistContext* c, Label title, double ea, int eb);
		/// Destruction
		virtual ~Erlang(){};

		/// Get next random number
		virtual double sample();

		/// Get parameter a
		double getA() {return a;}
		/// Get parameter b
		int getB() {return b;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	private:
		double  zyqab;
	protected:
		double a;
		int    b;
		int uses;
	};

	/** \class Rconst

		\ingroup rdist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Constant number generator

		\note Rconst from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Rconst provides a constant number through the
		Rdist interface.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Rconst: public Rdist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				Pointer to DistContext object

			\param d
				constant double returned by sample()
		*/
		Rconst(DistContext* c, Label title, double d) : Rdist(c, title), a(d), uses(0){};
		/// Destruction
		virtual ~Rconst(){};

		/// Get number
		virtual double sample(){uses++; return a;};

		/// Get parameter d
		double getA() {return a;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	private:
		const double a;
		int uses;
	};
}

#endif

