//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProtocolLayer.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in ProtocolLayer.h

	\sa ProtocolLayer.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <odemx/protocol/ProtocolLayer.h>
#include <odemx/protocol/ProtocolSimulation.h>
#include <odemx/protocol/ProtocolEntity.h>


#include <iostream>


using namespace odemx;


ProtocolLayer::ProtocolLayer( ProtocolSimulation* sim, Label l, ProtocolLayerObserver* o ):
	DefLabeledObject( sim, l ),
	Observable<ProtocolLayerObserver>( o ),
	env( sim )
	{
		env -> getLayerManager() -> addLayer( this );
		
		//trace
		getTrace() -> mark( this, markCreate ); 
		
		// observer
		_obsForEach( ProtocolLayerObserver, Create( this ) );
	}
	

ProtocolLayer::~ProtocolLayer()
	{
		//trace
		getTrace() -> mark( this, markDestroy );
		
		// observer
		_obsForEach( ProtocolLayerObserver, Destroy( this ) );
	}


void ProtocolLayer::addEntity( ProtocolEntity* e ) {
	
	//trace
	getTrace() -> mark( this, markAddEntity, dynamic_cast<TraceProducer*>(e) );
	
	// observer
	_obsForEach( ProtocolLayerObserver, AddEntity( this, e ) );
	
	entityList.push_back( e );
}


void ProtocolLayer::removeEntity( ProtocolEntity* e ) {

	//trace
	getTrace() -> mark( this, markRemoveEntity, dynamic_cast<TraceProducer*>(e) );
	
	// observer
	_obsForEach( ProtocolLayerObserver, RemoveEntity( this, e ) );	
	
	ProtocolEntityList::iterator iter;
	iter = find( entityList.begin(), entityList.end(), e ); 
	
	if ( iter != entityList.end() )
		entityList.erase( iter );
	else
		error( "ProtocolLayer::removeEntity(): protocol entity not in this layer" );
}


void ProtocolLayer::printEntityList() {
	
	std::cout << "..........................." << std::endl;

	ProtocolEntityList::iterator iter;
	for( iter = entityList.begin(); iter != entityList.end(); ++iter ) {
		std::cout << (*iter) -> getLabel() << std::endl;
	}
	
	std::cout << "..........................." << std::endl;
}


ProtocolEntityList& ProtocolLayer::getEntityList() {
	return this -> entityList;
}


ProtocolSimulation* ProtocolLayer::getSimulation() const {
	return env;
}



Trace* ProtocolLayer::getTrace() const {
	return dynamic_cast<Trace*>(env); 
}


void ProtocolLayerManager::addLayer( ProtocolLayer* layer ) {

	layerNameMap[ std::string(layer -> getLabel()) ] = layer;
	allLayersTopDown.push_back( layer );
}


ProtocolLayerList& ProtocolLayerManager::getLayerList() {
	return allLayersTopDown;
}


ProtocolLayer* ProtocolLayerManager::getLayerByLabel( Label l ) {
	
	return layerNameMap[ std::string(l) ];
}


const MarkTypeId ProtocolLayer::baseMarkId = 5200;

const MarkType ProtocolLayer::markCreate( "create layer", baseMarkId+1, typeid(ProtocolLayer) );
const MarkType ProtocolLayer::markDestroy( "destroy layer", baseMarkId+2, typeid(ProtocolLayer) );
const MarkType ProtocolLayer::markAddEntity( "add entity", baseMarkId+3, typeid(ProtocolLayer) );
const MarkType ProtocolLayer::markRemoveEntity( "remove entity", baseMarkId+4, typeid(ProtocolLayer) );

