//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file NetTopology.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::NetTopology

	\sa NetTopology.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#ifndef ODEMX_NETTOPOLOGY_INCLUDED
#define ODEMX_NETTOPOLOGY_INCLUDED

#include <map>
#include <vector>
#include <iostream>
#include <iomanip>  // for pretty printing

#include <odemx/protocol/ProtocolSimulation.h>
#include <odemx/random/Idist.h>
#include <odemx/util/Trace.h>


namespace odemx {

	// forward declaration
	class NetNode;
	class ProtocolSimulation;
	
	class NetTopologyObserver;
	
	/** \class ConnInfo

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %ConnInfo is a structure for maintaining data about node pair connections
		
		\sa NetNode, NetTopology

		ConnInfo is used to describe the connection between two nodes
		in terms of quality.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	struct ConnInfo {
		
		// must be without parameter for initialization of the NodeMatrix 
		/// Construction
		ConnInfo(): loss( 0 ), p( 0 ), nodeDistance( 1 ) {}
		
		/// Destruction
		~ConnInfo() { delete loss; }
		
		Idist* loss;		 ///< random variable for message loss
		double p;			 ///< explicit loss probability
		double nodeDistance; ///< distance between two nodes
							   // multiply with data for transmission delay
	};

	
	
	/** \var typedef std::vector< NetNode* > NodeVector;
	 
		\brief Vector containing pointers to NetNodes
	*/
	typedef std::vector< NetNode* > NodeVector;
	
	
	/** \var typedef std::vector< NetNode* > NodeVector;
		 
		\brief Matrix containing information about connection quality

		This matrix contains connection information between all
		node pairs. NetNodes are not to be deleted during simulation 
		as their address is used as map key.
	*/
	typedef std::map< NetNode* , std::map< NetNode*, ConnInfo > > NodeMatrix;



	/** \class NetTopology

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %NetTopology is a class for modeling the structure of a network
		
		\sa NetNode, ConnInfo

		A network's topology is described by nodes and the connections
		to their neighbors. This class provides the means to model the
		of a communications network and set the connection quality between
		nodes and their neighbors.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class NetTopology: public virtual TraceProducer,
					   public Observable<NetTopologyObserver> {

	public:	
		
		/**
			\brief Construction
			\param sim
				pointer to the ProtocolSimulation object
		*/
		NetTopology( ProtocolSimulation* sim, NetTopologyObserver* o = 0 );

		/// Destruction
		virtual ~NetTopology();
		
		/** 
			\name Node handling
			
			These functions provide access to the node structure of
			a network. Nodes can be registered or removed, and
			a list of all network nodes is available.
			
			@{ 
		*/
		/**
			\brief make a node known to the net
			\param n
				new NetNode
				
			The new node \p n will be added to the NodeMatrix, but by defaut
			it is not connected to any other nodes. The connection quality
			to its neighbors must be set afterwards.
		*/
		void registerNode( NetNode* n );
		

		/**
			\brief remove a node from the network structure
			\param n
				NetNode to be removed
				
			The node \p n will only be removed from the NodeMatrix. It will
			not be deleted by this function.
		*/
		void removeNode( NetNode* n );


		/**
			\brief get a vector listing all nodes currently in the network
			\return
				reference to a NodeVector containing all nodes
		*/
		NodeVector& getAllNodes();

		///	\brief  print current Matrix for debug purposes
		void printNodeMatrix();
		//@}

		
		/** 
			\name Connection handling
			
			These functions manage the connection quality between nodes.
			Connections can be set bidirectionally or with each direction
			separately. The NodeMatrix stores ConnInfo objects for both
			directions. The default connectivity is 0.
			
			@{
		*/
		/**
			\brief set connectivity for bi-directional connections
			\param n1
				pointer to first Node
			\param n2
				pointer to second Node
			\param p_both
				probability of message arrival

			The connectivity between the two nodes will be set to \p p_both 
			and a corresponding random variable will be initialized. Both 
			directions will get the same message arrival probability.
		*/
		void setConnection( NetNode* n1, NetNode* n2, double p_both );

		
		/**
			\brief set connectivity for bi-directional connections by label
			\param l1
				label of first Node
			\param l2
				label of second Node
			\param p_both
				probability of message arrival
				
			The connectivity between the two nodes will be set to \p p_both
			and a corresponding random variable will be initialized. Both 
			directions will get the same message arrival probability.
		*/
		void setConnection( Label l1, Label l2, double p_both );
		
		
		/**
			\brief set connectivity for each direction
			\param n1
				pointer to first Node
			\param n2
				pointer to second Node
			\param p_to
				probability of message arrival from \p n1 to \p n2
			\param p_from
				probability of message arrival from \p n2 to \p n1
				
			The connectivity between the two nodes will be set to \p p_to
			\p p_from respectively. Corresponding random variables will 
			be initialized. 
		*/
		void setConnection( NetNode* n1, NetNode* n2, double p_to, double p_from );
		

		/**
			\brief set connectivity for each direction
			\param l1
				pointer to first Node
			\param l2
				pointer to second Node
			\param p_to
				probability of message arrival from \p n1 to \p n2
			\param p_from
				probability of message arrival from \p n2 to \p n1
				
			The connectivity between the two nodes will be set to \p p_to
			\p p_from respectively. Corresponding random variables will 
			be initialized. 
		*/
		void setConnection( Label l1, Label l2, double p_to, double p_from );


		/**
			\brief cut connection for both directions
			\param n1
				pointer to first Node
			\param n2
				pointer to second Node
			\param both
				set \p false if only the direction \p n1 to \p n2 is to be disabled
				
			The connection quality between the two nodes will be reset to \p 0,
			and the corresponding random variables will be deleted. After
			calling this function, the two nodes will have no connection
			at all.
		*/
		void disconnect( NetNode* n1, NetNode* n2 );
		
		
		/**
			\brief check if two nodes are connected
			\param n1
				pointer to first Node
			\param n2
				pointer to second Node
			\return
				true if a connection was set between \p n1 and \p n2
		*/
		bool connected( NetNode* n1, NetNode* n2 );
		
		
		/// get the trace object
		Trace* getTrace() { return (Trace*)env; }
	
	private:

		/**
			\brief check if the given value is between 0 and 1
			\param p
				connection probability
			\return
				\p true if given probability is between 0 and 1
		*/
		bool validProbability( double p );
		

		/**
			\brief create a connection between nodes
			\param n1
				pointer to first Node
			\param n2
				pointer to second Node
			\param p
				connection probability
				
			This internal function creates a connection between two
			given nodes in direction \p n1 to \p n2 only. A random
			variable with the specified probability will be created.
		*/
		void makeConnection( NetNode* n1, NetNode* n2, double p );

		
		/**
			\brief change a connection between nodes
			\param n1
				pointer to first Node
			\param n2
				pointer to second Node
			\param p
				connection probability
				
			This internal function changes an existing connection between two
			given nodes in direction \p n1 to \p n2 only. A random
			variable with the specified probability will be created.
		*/
		void changeConnection( NetNode* n1, NetNode* n2, double p );

		
	protected:

		/// list of all nodes in the network
		NodeVector allNodes;
		
		/// two-dimensional structure for storing node reachability
		NodeMatrix connectivity;
		
	private:
	
		/// the simulation context
		ProtocolSimulation* env;
		
	public:
		
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetTopology calls.
			A TraceConsumer can use these constants to identify  
			trace events sent by NetTopology.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markRegisterNode;
		static const MarkType markRemoveNode;
		static const MarkType markMakeConnection;
		static const MarkType markChangeConnection;
		static const MarkType markDisconnect;
		
		static const TagId baseTagId;

		static const Tag tagFrom;
		static const Tag tagTo;
		static const Tag tagQuality;
		static const Tag tagNode;


		//@}
	};
	
	
	
	/** \interface NetTopologyObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for NetTopology specific simulation events.

		<!-- [\note {notes}]* -->

		\sa NetTopology, TransmissionMedium

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class NetTopologyObserver {
		
	public:
		/// Destructor
		virtual ~NetTopologyObserver() {}
		
		/// Register new NetNode
		virtual void onRegisterNode( NetTopology* sender, NetNode* n ) {}
		/// Remove an existing NetNode
		virtual void onRemoveNode( NetTopology* sender, NetNode* n ) {}
		/// Connect two NetNodes
		virtual void onMakeConnection( NetTopology* sender, NetNode* n1, NetNode* n2 ) {}
		/// Change connection of two NetNodes
		virtual void onChangeConnection( NetTopology* sender, NetNode* n1, NetNode* n2 ) {}
		/// Disable the connection between two NetNodes
		virtual void onDisconnect( NetTopology* sender, NetNode* n1, NetNode* n2 ) {}	
	};

	
}

#endif /* ODEMX_NETTOPOLOGY_INCLUDED */

