//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file SimpleProtocolMessage.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/06/10

	\brief Example for ODEMx protocol simulation techniques

	\sa SimpleProtocol.cpp SimpleProtocolEntities.h

	This example shows essential techniques for  
	implementing a protocol simulation with ODEMx.
	Protocols behavior is specified by employing the OSI
	layer model.

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/

/** 
	\example SimpleProtocolMessage.h

	With this example, the essential protocol simulation techniques of 
	ODEMx are introduced.

	Protocol models for ODEMx are based on the OSI layer model approach,
	where the protocol stack is separated into several layers, each with 
	a distinct function.

	A protocol simulation starts out with a ProtocolSimulation object, 
	which is basically a normal Simulation object, extended by functions
	to create and access the network topology, the transmission medium,
	and the protocol layers. 
	
	The network topology is described by NetNode objects and their
	connections, which are maintained by the class NetTopology. Derived 
	from this class is the TransmissionMedium, which also handles
	message transmissions between the network's nodes.
	
	A node itself has an input and an output port, which can be used 
	by the system environment to pass data to a node, and to register 
	successful message transmissions at a receiving node. The behavior
	of a node is modeled in its active components, so-called entities.
	
	A ProtocolEntity is a process representing a protocol layer function.
	Each node maintains exactly one entity per layer, which are 
	automatically created upon NetNode construction. Entities can receive
	input from the layer above or below and a behavior must be specified
	according to the data passed between these layers.

	In this example, a simple protocol simulation is defined, which shows
	the essential techniques to model and run a protocol simulation with
	ODEMx. Here, two protocol layers are modeled, one for the hardware
	part of a node, and one for the software part. The network contains 
	simple nodes, whose behavior is modeled by the definition of a
	HardwareEntity and a SoftwareEntity. The network's topology is
	definied in the simulation objects's initialization function. Lastly,
	messages between nodes are triggered by external events, which are also
	scheduled during the initialization phase.
	
	\sa SimpleProtocol.cpp SimpleProtocolEntities.h
*/


#ifndef PROTOCOLEXAMPLEMESSAGES_H_
#define PROTOCOLEXAMPLEMESSAGES_H_

#include <odemx/odemx.h>

#include <iostream>
#include <string>
#include <map>
#include <vector>


using namespace odemx;


//
// This file contains and implements the data types for messages
// in the SimpleProtocol example. 
// 
// Ports can only take PortElement*, the physical layer TransmissionMedium
// requires ProtocolMessage*, hence data types must be derived from these
// base classes.
//
// It is possible to specify a separate PDU type for each layer.
// 
// Each layer should fill the PDU information according to its 
// function. PDUs could be structured with different headers (PCI) 
// for each layer, according to the OSI model.
//


//
// Application layer: user data is sent to a node's interface
//
struct UserData: public PortElement {
 
	UserData( NetNode* s, NetNode* d, string str ): 
		source( s ),
		destination( d ),
		content( str ) 
		{} 
		
	const NetNode* source;
	const NetNode* destination;
	const string content;
};

//
// A vector containing the route is sent with messages.
// The routes are built by the SoftwareEntity of a node.
//
typedef std::vector< NetNode* > Route;
//
// The routing table is maintained by the SoftwareEntity 
// of each node. It contains destination nodes as keys
// and pointers to routes as values.
//
typedef std::map< NetNode*, Route* > RouteTable;

//
// For simplicity, besides UserData, there is only one message
// class, which can store different message types in a member.
// This means that one object can be passed back and forth 
// between nodes while only its type needs to be changed to 
// model route requests, route replies, acknowledgements, and 
// data packets.
//
class Message: public ProtocolMessage { // is a PortElement
public:
	//
	// The possible message types.
	//
	enum Type { UDAT, RREQ, RREP, ACK, DATA };
	
	Type messageType;
	int sequenceNumber;
	UserData* userData;
	NetNode* sendNode;
	NetNode* receiveNode;
	int routeMarker;
	int repeatedSends;
	Route* route;
	
	//
	// Each message must specify a type, a unique sequence number,
	// and the UserData it contains. The software part of a node
	// manages the routing, while the hardware part handles the
	// point-to-point transmissions and acknowledgements, including
	// setting of sender and receiver for each hop.
	//
	Message( Type type, int seqn, UserData* udat ):
		messageType( type ),
		sequenceNumber( seqn ),
		userData( udat ),
		sendNode( 0 ),
		receiveNode( 0 ),
		routeMarker( -1 ),
		repeatedSends( 0 )
		{}
	
	Type getMessageType() const { return messageType; }

	void addToRoute( NetNode* node ) {
		route -> push_back( node );
	}
	
	NetNode* getNextHop() {
		if ( (routeMarker > -1) && (routeMarker < (int)route -> size()) )	
			return (*route)[ routeMarker ];
		return 0;
	}
	
	Route* getRoute() {
		return route;
	}
	//
	// ALL of the functions below this point are required by the
	// interface ProtocolMessage. The TransmissionMedium uses
	// these functions to automate the whole transmission process.
	// 
	virtual NetNode* getSource() const { return (NetNode*)userData -> source; }
	virtual NetNode* getDestination() const { return (NetNode*)userData -> destination; }
	virtual NetNode* getSendNode() const { return sendNode; }
	virtual NetNode* getReceiveNode() const { return receiveNode; }
	//
	// Determines whether a Unicast or a Broadcast event will be scheduled.
	//
	virtual bool isUnicast() const { 
		return ( messageType == RREQ ) ? false : true; 
	}
	//
	// Is used to compute the transmission time of a message, i.e.
	// the time when the transmission event will be scheduled.
	//
	virtual SimTime getTransmissionDuration() const {
		return ( messageType == DATA ) ? (userData -> content).length() : 1;
	}
	//
	// This function is important when pointers are used in message
	// objects. If only a shallow copy were made, then both objects
	// would access the same object they are pointed to. This would
	// almost certainly cause conflicts.
	//
	virtual ProtocolMessage* deepCopy() {
		//
		// This will also just copy the UserData pointer, 
		// which is of no consequence as it is not changed anywhere.
		// The route, however must be properly separated by creating
		// a new Route object with the same content, which can then
		// be changed without side effects.
		//
		Message* msgCopy = new Message( *this );
		Route* routeCopy = new Route( *(route) );
		msgCopy -> route = routeCopy;
		return msgCopy;
	}
	//
	// This function is needed for debug purposes and for trace
	// marks sent by the TransmissionMedium and the transmission
	// events.
	//
	virtual std::string toString() {

		std::ostringstream out;
		out << "( ";
		
		switch ( messageType ) {
		case UDAT:
			out << "UDAT, "; break;
		case RREQ:
			out << "RREQ, "; break;
		case RREP:
			out << "RREP, "; break;
		case ACK:
			out << "ACK, "; break;
		case DATA:
			out << "DATA, "; break;
		}

		out << sequenceNumber << ", ";
		
		if ( userData -> source && userData -> destination ) { 
			out << userData -> source -> getLabel() << " to ";
			out << userData -> destination -> getLabel() << ", ";
		}
		else {
			out << "???, ";
		}
		
		if ( sendNode ) 
			out << sendNode -> getLabel() << " to ";
		
		if ( receiveNode ) 
			out << receiveNode -> getLabel() << ", ";
		else 
			out << " all, ";

		out << userData -> content;
		
		out << " )";
		return out.str();
	}
};

#endif /*PROTOCOLEXAMPLEMESSAGES_H_*/
