//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file InSort.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/22

	\brief insort algorithm for std::list<Process*> container

	<!-- [\sa <pair-file>] -->

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_INSORT_INCLUDED
#define ODEMX_INSORT_INCLUDED

#include <odemx/base/Process.h>

#include <list>
#include <algorithm>

namespace odemx {
	/**
		\brief insort for std::list<Process*>

		\param l
			reference to Process* container

		\param p
			pointer to new Process

		\param fifo
			use FIFO or LIFO strategy

		InSort is used for insertion of Process objects in std::list<Process*>
		container.
	*/
	inline void InSort(std::list<Process*>& l, Process* p, bool fifo = true) {
		std::list<Process*>::iterator i;

		// remove previous entry of T* p
		i = std::find(l.begin(), l.end(), p);
		if (i!=l.end())
			l.erase(i);

		// insert Process p in list
		if (fifo)
			for (i=l.begin(); i!=l.end() && !(*p<**i); ++i);
		else
			for (i=l.begin(); i!=l.end() && **i<*p; ++i);
		
		if (i==l.end())
			l.push_back(p); // get iterator position with rbegin()
		else
			l.insert(i, p); // store iterator i in p
	}


	/**
		\brief insort for std::list<Sched*>

		\param l
			reference to Sched* container

		\param p
			pointer to new Sched object

		\param fifo
			use FIFO or LIFO strategy

		This overloaded version of InSort is used for insertion of Sched 
		objects in std::list<Sched*> container. It was added to provide 
		InSort for the updated ExecutionList and other classes which might 
		be based on ordered Sched*-Lists
	 */

	inline void InSort(std::list<Sched*>& l, Sched* p, bool fifo = true) {
		std::list<Sched*>::iterator i;

		// remove previous entry of T* p
		i = std::find(l.begin(), l.end(), p);
		if (i!=l.end())
			l.erase(i);

		// insert Process p in list
		if (fifo)
			for (i=l.begin(); i!=l.end() && !(*p<**i); ++i);
		else
			for (i=l.begin(); i!=l.end() && **i<*p; ++i);
		if (i==l.end())
			l.push_back(p);
		else
			l.insert(i, p);
	}



	/**
		\brief insort for std::list<Process*>

		\param l
			reference to Process* container

		\param p
			pointer to new Process

		\param pr
			comparision functor

		\param fifo
			use FIFO or LIFO strategy

		InSort is used for insertion of Process objects in std::list<Process*>
		container by using a user-defined predicate as sorting criterion.
		
		\sa DefaultOrder, PriorityOrder
	*/
	template <typename Pred>
		inline void InSort(std::list<Process*>& l, Process* p, Pred pr, bool fifo = true) {
		std::list<Process*>::iterator i;

		// remove previous entry of T* p
		i = std::find(l.begin(), l.end(), p);
		if (i!=l.end())
			l.erase(i);

		// insert Process p in list
		if (fifo)
			for (i=l.begin(); i!=l.end() && !pr(*p,**i); ++i);
		else
			for (i=l.begin(); i!=l.end() && pr(**i,*p); ++i);
		if (i==l.end())
			l.push_back(p);
		else
			l.insert(i, p);
	}
}

#endif

