//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file LabeledObject.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/26

	\brief Declaration of odemx::Label, odemx::LabeledObject, odemx::LabelScope and odemx::DefLabeledObject

	\sa LabeledObject.cpp

	Label, LabeledObject, LabelScope and DefLabeledObject are used to provide
	short human readable text (labels) associated to objects.

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_LABELEDOBJECT_INCLUDED
#define ODEMX_LABELEDOBJECT_INCLUDED

#ifdef _MSC_VER
#pragma warning(disable : 4786)
#ifdef FORCE_STACK_IMPL
#pragma warning(disable : 4530)
#endif
#endif

#include <string>
#include <map>

namespace odemx {
	/** \var typedef const char* Label;
	 
		\brief Label type
	*/
	typedef const char* Label;

	/** \interface LabeledObject

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %LabeledObject is an interface that provides a
		label for an object.

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class LabeledObject {
	public:
		virtual ~LabeledObject(); ///< Destruction
		/**
			\brief get Label of object

			\return Label of object
		*/
		virtual Label getLabel() const = 0;
	};

	/** \class LabelScope

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %LabelScope provides a scope for a label.

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		LabelScope guarantees that every Label that is created
		in the scope is unique (in the scope).

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class LabelScope {
	public:
		/**
			\brief Construction

			\param defLabel
				default label

			\param defSpace
				default fillchar
		*/
		LabelScope(const char* defLabel="noname", char defSpace='_');

		/**
			\brief create a unique Label

			\param label
				basic part of the label, if left blank
				the default label is used

			createLabel builds a unique label. If there is already
			an equal label in this scope the new label gets a number.
		*/
		std::string createLabel(const char* label="");

	protected:
		const char* def;
		char spc;
		std::map<std::string, unsigned int> lookup;
	};

	/** \class DefLabeledObject

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %DefLabeledObject is a default implementation of the
		LabeledObject interface.

		<!-- [\note {notes}]* -->

		\sa LabeledObject LabelScope

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class DefLabeledObject : public virtual LabeledObject {
	public:
		/// Construction
		DefLabeledObject();

		DefLabeledObject(LabelScope* scope, const char* label);
		
		/**
			\brief get label

			\copydoc LabeledObject::getLabel()
		*/
		virtual Label getLabel() const;

	protected:
		/**
			\brief set label

			use setLabel to initialize the label
		*/
		void setLabel(LabelScope* scope, const char* label);

	private:
		/// Label
		std::string l;
	};
}

#endif

