//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Statistics.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/20

	\brief Implementation of classes in Statistics.h

	\sa Statistics.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/
#ifdef _MSC_VER
#pragma warning (disable : 4786)
#endif

#include <odemx/statistics/Statistics.h>
#include <odemx/util/ErrorHandling.h>

#include <cmath>
#include <cassert>

#include <sstream>
#include <string>
#include <algorithm>

using namespace std;
using namespace odemx;

Tab::Tab (Simulation* s, Label label/*=""*/) : StatisticObject(s), env(s)
{
	if (env==0) {
		// error: Tab was created without reference to Simulation.
		fatalError("Tab(); invalid argument: Simulation* s", -1);
	}

	// set label
	DefLabeledObject::setLabel(s, label);

	reset();
}

Tab::~Tab() {}


Count::Count (Simulation* s, Label title/*=""*/) : Tab(s, title) {
	reset();
}

Count::~Count() {};

void Count::report(Report* r) {
	assert(r);

	static const char* labels[] = {"Name", "Reset at", "User", "Value"};
	static const ColumnType types[] = {STRING, REAL, INTEGER, INTEGER};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Count Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << getObs() << getValue();
}

Sum::Sum (Simulation* s, Label title/*=""*/) : Tab(s, title) {
	reset();
}

Sum::~Sum() {}

void Sum::report(Report* r) {
	assert(r);

	static const char* labels[] = {"Name", "Reset at", "Uses", "Value"};
	static const ColumnType types[] = {STRING, REAL, INTEGER, INTEGER};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Sum Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << getObs() << getValue();

}

Tally::Tally (Simulation* s, Label title/*=""*/) : Tab(s, title) {
	reset();
}

Tally::~Tally() {}

void Tally::reset() {
	Tab::reset();
	sum = sumsq = min = max = 0.0;
}

void Tally::update (double v) {
	Tab::update();

	sum += v;
	sumsq += v*v;

	if (getObs() == 1) min = max = v;
	else if (v < min) min = v;
	else if (v > max) max = v;
}

double Tally::getDivergence() {
	if (getObs()<=1)
		return 0.0;

	double v=fabs(sumsq/getObs() - getMean()*getMean());
	return sqrt(v);
}

void Tally::report(Report* r) {
	assert(r);

	static const char* labels[] = {"Name", "Reset at", "Uses", "Size", "Min", "Max", "Mean", "Divergence"};
	static const ColumnType types[] = {STRING, REAL, INTEGER, INTEGER, REAL, REAL, REAL, REAL};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Tally Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << getObs() << getSize();
	*t << getMin() << getMax() << getMean() << getDivergence();
}

Accum::Accum (Simulation* s, Label title/*=""*/, Approx app/*=stepwise*/) : 
	Tab(s, title), approx(app) 
	{
		reset();
	}

Accum::~Accum() {}

void Accum::reset() {
	Tab::reset();
	sumt = sumsqt = min = max = lastv = 0.0;
	lasttime = env->getTime();
	reset_at = env->getTime();
}

void Accum::update (double v) {
	double now, span;

	Tab::update();

	now     = env->getTime();
	span    = now - lasttime;
	lasttime= now;

        if (getObs()>1) {
                if (approx==linear)
                {
                        double mean = (v+lastv)/2;
                        sumt += mean * span;
                        sumsqt += mean * mean * span;
                }
                else // approx == stepwise
                {
                        sumt   += lastv * span;
                        sumsqt += lastv * lastv * span;
                }
        }
	lastv   = v;

	if (getObs() == 1) min = max = v;
	else if (v < min) min = v;
	else if (v > max) max = v;
}

double Accum::getMean() const {
	double span, avg, t;

	t = env->getTime();
	span = t-getResetTime();
	t = t-lasttime;

	if (span == 0)
		return 0.0;

	avg = (sumt + lastv*t) / span;

	return avg;
}

double Accum::getDivergence() const {
	if (getObs()<=1)
		return 0.0;

	double span, avg, t, v;

	t = env->getTime();
	span = t-getResetTime();
	t = t-lasttime;

	avg = getMean();

	if (span == 0.0 )
		return 0.0;

	v = fabs( (sumsqt+lastv*lastv*t)/span - avg*avg );
	return sqrt(v);
}

void Accum::report(Report* r) {
	assert(r);

	static const char* labels[] = {"Name", "Reset at", "Uses", "Size", "Min", "Max", "Mean", "Divergence"};
	static const ColumnType types[] = {STRING, REAL, INTEGER, INTEGER, REAL, REAL, REAL, REAL};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Accum Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << getObs() << getSize();
	*t << getMin() << getMax() << getMean() << getDivergence();
}

Histo::Histo(Simulation* s, Label title, double low, double upp, int n)
 : Tally(s, title), data(n+2)
{
	lower=low;
	upper=upp;
	ncells=n;

	if (upper==lower) {
		upper=lower+100.0;
		warning("Histo: borders are equal; borders changed");
	}

	if (upper < lower) {
		std::swap(lower, upper);
		warning("Histo: upper borders < lower border; borders swapped");
	}

	if ((ncells < 1)) {
		warning("Histo: number of cells < 1; number increased");
		ncells=10;

		data.resize(ncells+2);
	}

	width = (upper-lower)/ncells;
	limit = ncells+1;
};

Histo::~Histo() {
}

void Histo::reset() {
	Tally::reset();
	for (int k = 0; k <= limit; k++) data[k] = 0;
}

void Histo::update(double v) {
	int cell;

	Tally::update(v);

	v -= lower;
	if (v<0.0) cell = 0;
	else {
		cell = (int)(floor(v/width))+1;
		if (cell>limit) cell = limit;
	}
	data[cell]++;
}

int Histo::maximumelem() {
	int k=0;
	int j;
	if (getObs()>0) {
		k=data[0];
		for (j=1; j<=limit; j++)
			if (data[j] > k) k=data[j];
	}
	return k;
}

void Histo::report(Report* r) {
	assert(r);

	// Summary (Tally)
	static const char* labels[] = {"Name", "Reset at", "Uses", "Low", "High", "Cells", "Size", "Min", "Max", "Mean", "Divergence"};
	static const ColumnType types[] = {STRING, REAL, INTEGER, REAL, REAL, INTEGER, INTEGER, REAL, REAL, REAL, REAL};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Histo Statistics - Summary", &def);
	assert(t);

	*t << getLabel() << getResetTime() << getObs() << lower << upper << ncells;
	*t << getSize() << getMin() << getMax() << getMean() << getDivergence();

	// Histogram
	static const char* labels2[] = {"Low", "High", "Count", "Percentage"};
	static const ColumnType types2[] = {REAL, REAL, INTEGER, REAL};
	static utTableDef def2(sizeof(labels2)/sizeof(char*), labels2, types2);
	std::ostringstream tableName;
	tableName << "Histogram: " << getLabel() << " [" << getSize() << " samples]" << std::ends;

	Table* t2=r->createTable(tableName.str(), &def2);
	assert(t2);

	std::vector<int>::const_iterator it=data.begin();
	if (*it!=0)
		*t2 << getMin() << lower << *it << ((*it)*100.0)/getObs();
	++it;

	for (int i=0; i<ncells; ++i, ++it)
		*t2 << (lower + i*width) << (lower + (i+1)*width) << *it << ((*it)*100.0)/getObs();

	if (*it!=0)
		*t2 << upper << getMax() << *it << ((*it)*100.0)/getObs();
}

Regress::Regress(Simulation* s, Label title)
 : Tab(s, title)
{
	reset();
}

Regress::~Regress() {}

void Regress::update(double vx, double vy) {
	Tab::update();

	x += vx;
	y += vy;
	xx = vx*vx + xx;
	xy = vx*vy + xy;
	yy = vy*vy + yy;
}

void Regress::reset() {
	Tab::reset();

	x = y = xx = xy = yy = 0.0;
}

double Regress::getXBar() {
	return x/getObs();
}

double Regress::getYBar() {
	return y/getObs();
}

double Regress::getDx() {
	return fabs(getObs()*xx - x*x);
}

double Regress::getDy() {
	return fabs(getObs()*yy - y*y);
}

double Regress::getV1() {
	// RES.ST.DEV
	double sd = sqrt((yy - getV3()*y - getV2()*xy)/(getObs() - 2));
	return sd;
}

double Regress::getV2() {
	// EST.REG.COEFF
	double a1 = (getObs()*xy - x*y)/getDx();
	return a1;
}

double Regress::getV3() {
	// INTERCEPT
	double a0 = (y*xx - x*xy)/getDx();
	return a0;
}

double Regress::getV4() {
	// ST.DEV.REG.COEFF
	return (getObs()*getV1()/sqrt((getObs()-2)*getDx()));
}

double Regress::getV5() {
	// CORR.COEFF
	double r2 = ((getObs()*xy - x*y)*(getObs()*xy - x*y))/(getDx()*getDy());
	return r2;
}

void Regress::report(Report* r) {
	assert(r);

	static const char* labels[] = {"Name", "Reset at", "Uses", "XBar", "YBar", "Dx", "Dy",
								   "RES.ST.DEV", "EST.REG.COEFF", "INTERCEPT", "CORR.COEFF"};
	static const ColumnType types[] = {STRING, REAL, INTEGER, REAL, REAL, REAL, REAL, REAL, REAL, REAL, REAL};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);
	assert(sizeof(labels)/sizeof(char*) == sizeof(types)/sizeof(ColumnType));

	Table* t=r->createTable("Regress Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << getObs();
	*t << getXBar() << getYBar() << getDx() << getDy();
	*t << getV1() << getV2() << getV3() << getV4() << getV5();
}

