//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Process.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/30

	\brief Implementation of classes in Process.h

	\sa Process.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/Process.h>
#include <odemx/base/Event.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/ErrorHandling.h>

#include <list>
#include <string>

#include <odemx/Debug.h>

using namespace odemx;
using namespace std;

Process::Process(Simulation* s, Label l, ProcessObserver* o) :
	Sched(PROCESS),
	Coroutine(l, s, o),
	Observable<ProcessObserver>(o),
	processState(CREATED),
	p(0.0),
	t(0.0),
	env(s),
	returnValue(0),
	q(0),
	q_int(0.),
	q_out(0.),
	validReturn(false),
	interrupted(false),
	interrupter(0),
	alerted(false),
	alerter(0),
	internalMemVec( false )
	{
		if (s==0) {
			// error: Process was created without reference to simulation.
			fatalError("Process(); invalid argument: Simulation* s", -1);
		}
	
		// trace
		getTrace()->mark(this, markCreate);
	
		// observer
		_obsForEach(ProcessObserver, Create(this));
	
		env->setProcessAsCreated(this);
	}

Process::~Process() {

	// trace
	getTrace()->mark(this, markDestroy);

	// observer
	_obsForEach(ProcessObserver, Destroy(this));

	if (!env->isFinished()) {
		// warning: Process should not be destroyed while simulation is running.
		warning("~Process(); simulation not finished");
	}
}

Process::ProcessState Process::getProcessState() const {
	return processState;
}

void Process::activate() {
	// trace
	getTrace()->mark(this, markActivate, getPartner());

	// observer
	_obsForEach(ProcessObserver, Activate(this));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();
	
	// reset Memo::alert state
	resetAlert();

	// scheduling
	setExecutionTime(getCurrentTime());
	env->insertSched(this);

	// continue simulation
	env->compSimulation();
}

void Process::activateIn(SimTime t) {
	// trace
	traceActivateIn(t);

	// observer
	_obsForEach(ProcessObserver, ActivateIn(this, t));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	setExecutionTime(getCurrentTime() + t);
	env->insertSched(this);

	// continue simulation
	env->compSimulation();
}

void Process::activateAt(SimTime t) {
	// trace
	traceActivateAt(t);

	// observer
	_obsForEach(ProcessObserver, ActivateAt(this, t));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	setExecutionTime(t);
	env->insertSched(this);

	// continue simulation
	env->compSimulation();
}

void Process::activateBefore(Process* p) {
	// trace
	traceActivateBefore(p);

	// observer
	_obsForEach(ProcessObserver, ActivateBefore(this, p));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	env->insertSchedBefore(this, p);

	// continue simulation
	env->compSimulation();
}

void Process::activateAfter(Process* p) {
	// trace
	traceActivateAfter(p);

	// observer
	_obsForEach(ProcessObserver, ActivateAfter(this, p));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	env->insertSchedAfter(this, p);

	// continue simulation
	env->compSimulation();
}

void Process::hold() {
	// trace
	getTrace()->mark(this, markHold, getPartner());

	// observer
	_obsForEach(ProcessObserver, Hold(this));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	setExecutionTime(getCurrentTime());
	env->addSched(this);

	// continue simulation
	env->compSimulation();
}

void Process::holdFor(SimTime t) {
	// trace
	traceHoldFor(t);

	// observer
	_obsForEach(ProcessObserver, HoldFor(this, t));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	setExecutionTime(getCurrentTime() + t);
	env->addSched(this);

	// continue simulation
	env->compSimulation();
}


void Process::holdUntil(SimTime t) {
	// trace
	traceHoldUntil(t);

	// observer
	_obsForEach(ProcessObserver, HoldUntil(this, t));

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// reset interrupt state
	resetInterrupt();

	// reset Memo::alert state
	resetAlert();

	// scheduling
	setExecutionTime(t);
	env->addSched(this);

	// continue simulation
	env->compSimulation();
}


// m0 always has to be given, else no matching call
Memo* Process::wait( Memo* m0, Memo* m1, Memo* m2, 
					 Memo* m3, Memo* m4, Memo* m5 ) {
	
	MemoVector* memvec = new MemoVector;
	internalMemVec = true;
	
	memvec -> reserve( 6 );
	memvec -> push_back( m0 );
	
	// add more parameters to vector, if they are given
	if ( m1 ) memvec->push_back( m1 );
	if ( m2 ) memvec->push_back( m2 );
	if ( m3 ) memvec->push_back( m3 );
	if ( m4 ) memvec->push_back( m4 );
	if ( m5 ) memvec->push_back( m5 );

	return wait( memvec );
}


Memo* Process::wait( MemoVector* memvec ) {

	if ( ( !memvec ) || ( memvec -> size() == 0 ) ) {
		
		// warning no alerter given
		warning( "Process::wait(): No alerter given" );
		return 0;
	}

	// trace
	traceWait( memvec );

	
	// observer
	_obsForEach( ProcessObserver, Wait( this, memvec ) );

	// before suspending, check availability of resources
	MemoVector::iterator iter;
		for ( iter = memvec -> begin(); iter != memvec -> end(); ++iter ) {

		// check Memo object availability
		if ( (*iter) -> available() ) {

			Memo* theAvailable = (*iter);

			// trace
			getTrace() -> mark( this, markAvailable, theAvailable -> getLabel() );

			// warn if Timer is not set
			if ( theAvailable -> getMemoType() == Memo::TIMER )
				warning( (string("Process::wait(): Timer available due to not being set: ") + theAvailable -> getLabel()).c_str() );

			// clean up
			if ( internalMemVec ) { 
				delete memvec;
				internalMemVec = false;
			}

			return theAvailable;
		}
	}

	// trace
//	traceWait( memvec );

	// since none are available, register this process with the given Memo objects
	for ( iter = memvec -> begin(); iter != memvec -> end(); ++iter ) {

		(*iter) -> remember( this );
	}
	
	// the alert mechanism is normally only used in wait()
	// reset alerter and alerted anyway, if set somewhere else
	resetAlert();

	// suspend process
	sleep();

	// ************* beyond this point, the process was awakened ************

	// check for Memo alert
	if ( isAlerted() ){
		
		Memo* currentAlerter = getAlerter();
		
		for ( iter = memvec -> begin(); iter != memvec -> end(); ++iter ) {
		
			if ((*iter) != alerter) // alerter has already removed this process
				(*iter) -> forget(this);
		}
		
		// reset so outdated alerted/alerter cannot be used elsewhere by accident
		resetAlert();
		
		if ( internalMemVec ) { 
			delete memvec;
			internalMemVec = false;
		}
		
		return currentAlerter;
	}
	// the sleep() period can be interrupted by other processes
	else if ( isInterrupted() ) {
		
		// remove this from all Memos
		for ( iter = memvec->begin(); iter != memvec->end(); ++iter ) {
		
			(*iter) -> forget(this);
		}
		
		warning( "Process::wait(): another Process called interrupt to end wait()" );

		if ( internalMemVec ) { 
			delete memvec;
			internalMemVec = false;
		}		
		// when wait returns 0, a Process should check for interruption 
		return 0;
	}
	else 
		error( "Process::wait(): Process awakened without alert or interrupt" );
	
	return 0;
}


void Process::alertProcess( Memo* theAlerter ) {
	
	// trace
	getTrace() -> mark( this, markAlert );
	
	// observer
	_obsForEach( ProcessObserver, Alert( this, theAlerter ) );

	alerted = true;

	alerter = theAlerter;

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// scheduling
	setExecutionTime(getCurrentTime());
	env->insertSched(this);
}


bool Process::isAlerted() const {
	return alerted;	
}


Memo* Process::getAlerter() const {
	return alerter;
}


void Process::resetAlert() {
	alerted = false;
	alerter = 0;
}


void Process::interrupt() {
	if (this == getCurrentProcess()) {
		// ignore self interrupt
		return;
	}

	if (isInterrupted()) {
		// Warning: Process already interrupted
		warning("interrupt(); Process already interrupted");
	}

	// trace
	getTrace()->mark(this, markInterrupt, getPartner());

	// observer
	_obsForEach(ProcessObserver, Interrupt(this));

	// Set interrupt data
	interrupted = true;
	interrupter = getCurrentProcess();

	// state management
	if (!setProcessState(RUNNABLE))
		return;

	// scheduling
	setExecutionTime(getCurrentTime());
	env->insertSched(this);

	// continue simulation
	env->compSimulation();
}


void Process::sleep() {
	// trace
	getTrace()->mark(this, markSleep, getPartner());

	// observer
	_obsForEach(ProcessObserver, Sleep(this));

	// state management
	if (!setProcessState(IDLE))
		return;

	// scheduling
	env->removeSched(this);

	// continue simulation
	env->compSimulation();
}

void Process::cancel() {
	// trace
	getTrace()->mark(this, markCancel, getPartner());

	// observer
	_obsForEach(ProcessObserver, Cancel(this));

	if (setProcessState(TERMINATED)) {
		// scheduling
		env->removeSched(this);

		// continue simulation
		env->compSimulation();
	}
}

Process* Process::getCurrentProcess() {
	return env->getCurrentProcess();
}

SimTime Process::getCurrentTime() {
	return env->getTime();
}

Simulation* Process::getSimulation() {
	return env;
}

int Process::getReturnValue() const {
	if (!validReturn) {
		// Warning: return value is not valid
		warning("getReturnValue(); return value is not valid");
	}

	return returnValue;
}


void Process::execute() {

	if (processState!=RUNNABLE && processState!=CURRENT) {
		// error: Process has to be RUNNABLE or CURRENT to be executed.
		fatalError("execute(); process is not RUNNABLE or CURRENT", -1);
	}

	// trace
	getTrace()->mark(this, markExecute, getPartner());

	// observer
	_obsForEach(ProcessObserver, Execute(this));

	// old currentSched value is needed in getPartner()
	setProcessState(CURRENT);

	// switch to Coroutine
	switchTo();
}

bool Process::setProcessState(ProcessState newState) {
	if (processState==TERMINATED) {
		// error: Process already terminated
		error("setProcessState(); changing state of terminated process is impossible");
		return false;
	}

	if (newState==CREATED) {
		// error: Process already created
		error("setProcessState(); Process already created");
		return false;
	}

	// trace
	getTrace()->mark(this, markChangeProcessState, newState, processState);

	// observer
	_obsAForEach(ProcessObserver, ProcessState, processState, newState);

	processState = newState;

	switch (processState) {
	case CREATED:
		env->setProcessAsCreated(this);
		break;
	case RUNNABLE:
		env->setProcessAsRunnable(this);
		break;
	case CURRENT:
		env->setProcessAsCurrent(this);
		env -> setCurrentSched( this );
		break;
	case IDLE:
		env->setProcessAsIdle(this);
		break;
	case TERMINATED:
		env->setProcessAsTerminated(this);
		break;
	}

	return true;
}

void Process::start() {
	returnValue = main();

	// Process finished [returned from main()]
	if (setProcessState(TERMINATED))
		env->removeSched(this);

	// The return value is valid now
	validReturn=true;

	// continue simulation
	env->compSimulation();
}

Priority Process::getPriority() const {
	return p;
}

Priority Process::setPriority(Priority newPriority) {
	Priority oldPriority = p;
	p = newPriority;

	// trace
	getTrace()->mark(this, markChangePriority, newPriority, oldPriority);

	// observer
	_obsAForEach(ProcessObserver, Priority, oldPriority, newPriority);

	// setPriority() influences ExecutionList and ProcessQueue
	if( getProcessState() == RUNNABLE || getProcessState() == CURRENT )
		env->inSort(this);
	
	if (q!=0) {
		q->inSort(this);
		activate();
	}

	// continue simulation (process change possible)
	env->compSimulation();

	return oldPriority;
}

SimTime Process::getExecutionTime() const {
	if (processState!=RUNNABLE &&
		processState!=CURRENT)
		return 0.0;

	return t;
}

SimTime Process::setExecutionTime( SimTime time ) {
	SimTime oldTime = t;
	t = time;

	// trace
	getTrace()->mark(this, markChangeExTime, time, oldTime);

	// observer
	_obsAForEach(ProcessObserver, ExecutionTime, oldTime, t);

	return oldTime;
}

void Process::enqueue(ProcessQueue* nq) {
	if (q!=0) {
		// error: Process is already listed in ProcessQueue.
		fatalError("enqueue(); Process already listed in a ProcessQueue", -1);
	}

	q=nq;
	q_int=getCurrentTime();
}

void Process::dequeue(ProcessQueue* nq) {
	if (q!=nq) {
		// error: Process dequeued() by wrong ProcessQueue
		error("dequeue(); Process dequeued() by wrong ProcessQueue");
		return;
	}

	q=0;
	q_out=getCurrentTime();
}


TraceProducer* Process::getPartner() {

	if ( env -> getCurrentSched() != 0 ) {
		
		return dynamic_cast< TraceProducer* >(env -> getCurrentSched());
	}
	else {
		
		return env;
	}
}



inline void Process::traceActivateIn(SimTime t) {
	getTrace()->beginMark(this, markActivateIn);
	getTrace()->addTag(tagCurrent, getPartner());
	getTrace()->addTag(tagRelSimTime, t);
	getTrace()->endMark();
}

inline void Process::traceActivateAt(SimTime t) {
	getTrace()->beginMark(this, markActivateAt);
	getTrace()->addTag(tagCurrent, getPartner());
	getTrace()->addTag(tagAbsSimTime, t);
	getTrace()->endMark();
}

void Process::traceActivateBefore(Process* p) {
	getTrace()->beginMark(this, markActivateBefore);
	getTrace()->addTag(tagCurrent, getPartner());
	getTrace()->addTag(tagPartner, p);
	getTrace()->endMark();
}

void Process::traceActivateAfter(Process* p) {
	getTrace()->beginMark(this, markActivateAfter);
	getTrace()->addTag(tagCurrent, getPartner());
	getTrace()->addTag(tagPartner, p);
	getTrace()->endMark();
}

void Process::traceHoldFor(SimTime t) {
	getTrace()->beginMark(this, markHoldFor);
	getTrace()->addTag(tagCurrent, getPartner());
	getTrace()->addTag(tagRelSimTime, t);
	getTrace()->endMark();
}

void Process::traceHoldUntil(SimTime t) {
	getTrace()->beginMark(this, markHoldUntil);
	getTrace()->addTag(tagCurrent, getPartner());
	getTrace()->addTag(tagAbsSimTime, t);
	getTrace()->endMark();
}

void Process::traceWait( MemoVector* memvec ) {
	getTrace() -> beginMark( this, markWait );
	MemoVector::iterator iter;
	for ( iter = memvec -> begin(); iter != memvec -> end(); ++iter ) {
		getTrace() -> addTag( tagPartner, dynamic_cast<TraceProducer*>(*iter) );
	}
	getTrace() -> endMark();
}


const MarkTypeId Process::baseMarkId=1000;

const MarkType Process::markCreate("create", baseMarkId+1, typeid(Process));
const MarkType Process::markDestroy("destroy", baseMarkId+2, typeid(Process));

const MarkType Process::markActivate("activate", baseMarkId+10, typeid(Process));
const MarkType Process::markActivateIn("activateIn", baseMarkId+11, typeid(Process));
const MarkType Process::markActivateAt("activateAt", baseMarkId+12, typeid(Process));
const MarkType Process::markActivateBefore("activateBefore", baseMarkId+13, typeid(Process));
const MarkType Process::markActivateAfter("activateAfter", baseMarkId+14, typeid(Process));
const MarkType Process::markHold("hold", baseMarkId+15, typeid(Process));
const MarkType Process::markHoldFor("holdFor", baseMarkId+16, typeid(Process));
const MarkType Process::markHoldUntil("holdUntil", baseMarkId+17, typeid(Process));
const MarkType Process::markInterrupt("interrupt", baseMarkId+18, typeid(Process));
const MarkType Process::markSleep("sleep", baseMarkId+19, typeid(Process));
const MarkType Process::markCancel("cancel", baseMarkId+20, typeid(Process));

const MarkType Process::markExecute("execute", baseMarkId+21, typeid(Process));

const MarkType Process::markWait("wait", baseMarkId+22, typeid(Process));
const MarkType Process::markAvailable("found available", baseMarkId+23, typeid(Process));
const MarkType Process::markAlert("alert", baseMarkId+24, typeid(Process));

const MarkType Process::markChangeProcessState("changeState", baseMarkId+30, typeid(Process));
const MarkType Process::markChangePriority("changePriority", baseMarkId+31, typeid(Process));
const MarkType Process::markChangeExTime("changeExTime", baseMarkId+32, typeid(Process));

const TagId Process::baseTagId = 1000;

const Tag Process::tagCurrent("current");
const Tag Process::tagAbsSimTime("absolute Time");
const Tag Process::tagRelSimTime("relative Time");
const Tag Process::tagPartner("partner");
