//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file HtmlTrace.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/06/09

	\brief Implementation of classes in HtmlTrace.h

	\sa HtmlTrace.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/HtmlTrace.h>
#include <odemx/util/Version.h>
#include <odemx/util/ErrorHandling.h>

#include <fstream>
#include <cassert>
#include <cstring>

using namespace std;
using namespace odemx;

HtmlTrace::HtmlTrace(Simulation* s, const char* fileName/*= 0*/) :
	sim(s),
	manageOut(true),
	pause(false),
	active(false),
	out(&cout),
	lastTime(-1.0),
	lastSender(0),
	colorSelect(0)
{
	assert(sim!=0);

	if (fileName==0)
		manageOut = false;
	else {
		out = new ofstream(fileName);
		if (out==0 || !(*out)) {
			// Error: cannot open file
			error("HtmlTrace cannot open file; sending output to stdout.");
			manageOut = false;
			out = &cout;
		}
	}
}

HtmlTrace::HtmlTrace(Simulation* s, std::ostream& os) :
	sim(s), manageOut(false), pause(false), active(false), out(&os)
{
	assert(sim!=0);

	if (!(*out)) {
		// Error: invalid ostream
		error("HtmlTrace invalid ostream; sending output to stdout.");
		out = &cout;
	}
}

HtmlTrace::~HtmlTrace() {
	if (manageOut) {
		static_cast<ofstream*>(out)->close();
		delete out;
	}
}

//
// Simple trace marks
//
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, const char* comment/*= 0*/) {
	pass(m, sender);
	if (skipOutput()) return;

	beginLine(sender, m);
	endLine(comment);
}

void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* partner, const char* comment/*= 0*/) {
	pass(m, sender);
	if (skipOutput()) return;

	beginLine(sender, m);
	table(1);
	*out << "<tr>";
	cell(1,0);
	*out << "Partner" << "</td>";
	cell(1,1);
	*out << partner->getLabel() << "</td></tr></table>";
	endLine(comment);
}

void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment/*= 0*/) {
	pass(m, sender);
	if (skipOutput()) return;

	beginLine(sender, m);
	table(1);
	*out << "<tr>";
	cell(1,0);
	*out << "1. Partner" << "</td>";
	cell(1,1);
	*out << firstPartner->getLabel() << "</td></tr>";

	*out << "<tr>";
	cell(1,0);
	*out << "2. Partner" << "</td>";
	cell(1,1);
	*out << secondPartner->getLabel() << "</td></tr></table>";
	endLine(comment);
}

void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, char newValue, char oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, short newValue, short oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, int newValue, int oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, long newValue, long oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, float newValue, float oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, double newValue, double oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}

void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned char newValue, unsigned char oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned short newValue, unsigned short oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned int newValue, unsigned int oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}
void HtmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned long newValue, unsigned long oldValue, const char* comment/*= 0*/) {
	writeLine(sender, m, newValue, oldValue, comment);
}

//
// Complex trace marks
//
void HtmlTrace::beginMark(const TraceProducer* sender, const MarkType* m) {
	pass(m, sender);
	if (skipOutput()) return;

	beginLine(sender, m);
	table(1);
}

void HtmlTrace::endMark(const char* comment /*=0*/) {
	if (skipOutput()) return;

	*out << "</table>";
	endLine(comment);
}

// Tags
void HtmlTrace::beginTag(Tag t) {
	if (skipOutput()) return;

	*out << "<tr>";
	cell(1,0);
	if (t.getName()!=Tag::NONAME)
		*out << translate(t.getName());
	else
		*out << t.getId();
	*out << "</td>";
	cell(1,1);
	table(2);
}

void HtmlTrace::endTag(Tag t) {
	if (skipOutput()) return;

	*out << "</table></td><tr>";
}

void HtmlTrace::addTag(Tag t, bool value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, char value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, short value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, int value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, long value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, float value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, double value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, const char* value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, const TraceProducer* value) {
	writeValue(t, value);
}

void HtmlTrace::addTag(Tag t, unsigned char value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, unsigned short value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, unsigned int value) {
	writeValue(t, value);
}
void HtmlTrace::addTag(Tag t, unsigned long value) {
	writeValue(t, value);
}


//
// trace control
//
// HtmlTrace starts a new trace
void HtmlTrace::startTrace() {
	active=true;

	// write DOCTYPE
	*out << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\">" << endl;

	// write HTML-Header
	*out << "<html>" << endl << "<title>" << endl;
	*out << "HtmlTrace for simulation: " << translate(sim->getLabel()) << ' ' << "at simtime: " << sim->getTime() << endl;
	*out << "</title>" << endl;

	// begin body
	*out << "<body>" << endl;

	// begin Table
	table(0);
	*out << endl;
	*out << "<thead>" << endl;

	// Simulation name and simtime
	*out << "<tr><th colspan=\"3\" align=\"left\"><h1> Simulation: ";
	*out << translate(sim->getLabel());
	*out << "</h1></th><th align=\"right\"><h1>SimTime:</h1></th>";
	*out << "<th align=\"right\"><h1>" << sim->getTime() << "</h1></th></tr>" << endl;

	// HtmlReport and ODEMx version
	*out << "<tr><th colspan=\"3\" align=\"left\"><h2>" << "HtmlTrace";
	*out << "</h2></th><th align=\"right\"><h2>ODEMx Version:</h2></th>";
	*out << "<th align=\"right\"><h2>" << translate(Version::getString()) << "</h2></th></tr>" << endl;

	// Filter
	*out << "<tr><th align=\"left\"><h2>" << "Filter:" << "</h2></th>";
	*out << "<th colspan=\"4\" align=\"left\"><h2>\"" << translate(getFilter()) << "\"</h2></th></tr>" << endl;

	// Table head and foot
	*out << "<tr height=\"20\"><th colspan=\"5\"></th></tr>" << endl;

	*out << "<tr bgcolor=\"darkturquoise\"><th>Time</th><th>Sender</th><th>Event</th><th>Details</th><th>Comment</th></tr>" << endl;
	*out << "</thead><tfoot>" << endl;
	*out << "<tr bgcolor=\"darkturquoise\"><th>Time</th><th>Sender</th><th>Event</th><th>Details</th><th>Comment</th></tr>" << endl;
	*out << "</tfoot><tbody>" << endl;
}

// XmlTrace stops the trace
void HtmlTrace::stopTrace() {
	active=false;

	// end Table
	*out << "</tbody></table>" << endl;

	// close body
	*out << "</body>" << endl;

	// close Html
	*out << "</html>" << endl;
}

// This call signals a break in trace: A TraceConsumer should not 
// produce output after this call is sent,
// but await further marks for updating internal data if required.
void HtmlTrace::pauseTrace() {
	pause = true;

	*out << "<tr>";
	cell(0,0);
	*out << sim->getTime() << "</td>";
	cell(0,1);
	*out << "BREAK</td><td></td><td></td><td></td></tr>" << endl;
}

// This call signals the end of a break in trace: A TraceConsumer should update its output if required,
// and report following marks.
void HtmlTrace::continueTrace() {
	pause = false;

	*out << "<tr>";
	cell(0,0);
	*out << sim->getTime() << "</td>";
	cell(0,1);
	*out << "CONTINUE</td><td></td><td></td><td></td></tr>" << endl;
}

const char* HtmlTrace::translate(const char* s) {
	string tmp=(s ? s : "");
	return translate(tmp);
}

const char* HtmlTrace::translate(const std::string& s) {
	static string ret;
	string::const_iterator i;

	// prepare ret
	ret.erase();

	// replace forbidden characters
	for (i=s.begin(); i!=s.end(); ++i) {
		switch (*i) {
		case '<':
			ret+="&lt;";
			break;
		case '>':
			ret+="&gt;";
			break;
		default:
			ret+=*i;
		}
	}
	ret+='\0';

	return ret.c_str();
}

void HtmlTrace::beginLine(const TraceProducer* sender, const MarkType* m) {
	*out << "<tr valign=\"top\">";

	// change colour
	if (sim->getTime()!=lastTime)
		colorSelect = (colorSelect+1)%2;

	// write time
	cell(0, 0);
	if (sim->getTime()!=lastTime) {
		*out << sim->getTime();
		lastTime = sim->getTime();
	}
	*out << "</td>";

	// write sender
	cell(0, 1);
	if (sender!=lastSender) {
		*out << /*translate(sender->getType().name()) << " :: " <<*/ translate(sender->getLabel());
		// *out << translate(m->getScope().name()) << " :: ";
		lastSender=sender;
	}
	*out << "</td>";

	// write event
	cell(0, 2);
	if (m->getName()!=MarkType::NONAME)
		*out << translate(m->getName());
	else
		*out << m->getId();
	*out << "</td>";
	cell(0, 3);
}

void HtmlTrace::endLine(const char* comment) {
	// write comment
	*out << "</td>";
	cell(0, 4);
	*out << translate(comment) << "</td></tr>" << endl;
}

void HtmlTrace::writeLine(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment) {
	pass(m, sender);
	if (skipOutput())
		return;

	beginLine(sender, m);
	table(1);
	*out << "<tr>";
	cell(1,0);
	*out << "old Value" << "</td>";
	cell(1,1);
	*out << (oldValue ? "true" : "false") << "</td></tr>";

	*out << "<tr>";
	cell(1,0);
	*out << "new Value" << "</td>";
	cell(1,1);
	*out << (newValue ? "true" : "false") << "</td></tr></table>";

	endLine(comment);
}

void HtmlTrace::writeLine(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment) {
	pass(m, sender);
	if (skipOutput())
		return;

	beginLine(sender, m);
	table(1);
	*out << "<tr>";
	cell(1,0);
	*out << "old Value" << "</td>";
	cell(1,1);
	*out << translate(oldValue) << "</td></tr>";

	*out << "<tr>";
	cell(1,0);
	*out << "new Value" << "</td>";
	cell(1,1);
	*out << translate(newValue) << "</td></tr></table>";

	endLine(comment);
}

void HtmlTrace::writeValue(Tag t, bool value) {
	if (skipOutput()) return;

	*out << "<tr>";
	cell(2,0);
	if (t.getName()!=Tag::NONAME)
		*out << translate(t.getName());
	else
		*out << t.getId();
	*out << "</td>";
	cell(2,1);
	*out << (value ? "true" : "false") << "</td></tr>";
}

void HtmlTrace::writeValue(Tag t, const char* value) {
	if (skipOutput()) return;

	*out << "<tr>";
	cell(2,0);
	if (t.getName()!=Tag::NONAME)
		*out << translate(t.getName());
	else
		*out << t.getId();
	*out << "</td>";
	cell(2,1);
	*out << translate(value) << "</td></tr>";
}

void HtmlTrace::writeValue(Tag t, const TraceProducer* value) {
	if (skipOutput()) return;

	*out << "<tr>";
	cell(2,0);
	if (t.getName()!=Tag::NONAME)
		*out << translate(t.getName());
	else
		*out << t.getId();
	*out << "</td>";
	cell(2,1);
	*out << translate(value->getLabel()) << "</td></tr>";
}

void HtmlTrace::table(int i) {
	*out << "<table ";
	switch (i) {
	case 0:
		// global table
		*out << "width=\"800\" border=\"1\" rules=\"rows\" frame=\"hsides\"";
		break;
	case 1:
		// detail table
		*out << "width=\"100%\" border=\"1\" rules=\"all\" frame=\"vsides\"";
		break;
	case 2:
		// inner table
		*out << "width=\"100%\" border=\"1\" rules=\"all\"";
		break;
	}
	*out << '>';
}

void HtmlTrace::cell(int i, int j) {
	*out << "<td ";
	switch (i) {
	case 0:
		// global table
		switch (j) {
		case 0:
			// Time
			*out << "width=\"50\" align=\"right\" bgcolor=\"" << colors[colorSelect] << '\"';
			break;
		case 1:
			// Sender
			*out << "width=\"150\"";
			break;
		case 2:
			// Event
			*out << "width=\"150\" bgcolor=\"" << colors[colorSelect] << '\"';
			break;
		case 3:
			// Detail
			break;
		case 4:
			// Comment
			*out << "width=\"150\" bgcolor=\"" << colors[colorSelect] << '\"';
			break;
		}
		break;
	case 1:
	case 2:
		// detail table
		// inner table
		switch (j) {
		case 0:
			// name
			*out << "width=\"50%\" align=\"left\"";
			break;
		case 1:
			// value
			*out << "width=\"50%\" align=\"right\"";
			break;
		}
		break;
	}
	*out << '>';
}

void HtmlTrace::setFilter(const char* f) {
	if (active && !pause) {
		// Error: HtmlTrace can change filter only if in pause or before startTrace()
		error("HtmlTrace can change filter only if in pause or before startTrace()");
		return;
	}

	TraceFilter::setFilter(f);

	if (active) {
		*out << "<tr>";
		cell(0,0);
		*out << sim->getTime() << "</td>";
		cell(0,1);
		*out << "NEW FILTER</td>";
		*out << "<td bgcolor=\"lightgreen\" colspan=\"3\">";
		*out << translate(getFilter()) << "</td></tr>" << endl;
	}
}

const char* HtmlTrace::colors[] = {"azure", "beige"};
