//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Memo.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in Memo.h

	\sa Memo.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <odemx/base/Process.h>
#include <odemx/synchronization/Memo.h>
#include <algorithm>

#include <odemx/util/ErrorHandling.h>
#include <odemx/Debug.h>

using namespace odemx;


Memo::Memo( Simulation* sim, MemoType t, MemoObserver* mo ):
	Observable<MemoObserver>( mo ),
	env( sim ),
	type( t )
	{
		// observer
		_obsForEach(MemoObserver, Create(this));
	}


Memo::~Memo() 
	{
		// observer
		_obsForEach(MemoObserver, Destroy(this));
	}


bool Memo::remember( Sched* newObject ) {
	
	// check if element is already memorized
	std::list< Sched* >::iterator iter;
	
	iter = find( memoryList.begin(), memoryList.end(), newObject );
	
	if ( iter != memoryList.end() ) {
	
		warning( "Memo::remember(): object already stored in memoryList" );
		return false;
	}
	
	// trace
	getTrace() -> mark( this, markRemember, dynamic_cast<TraceProducer*>(newObject) );
	
	// observer
	_obsForEach( MemoObserver, Remember( this, newObject ) );
	
	
	memoryList.push_back( newObject );
	return true;
}


bool Memo::forget( Sched* rememberedObject ) {
	
	// check if rememberedObject is in memoryList
	std::list<Sched*>::iterator iter;
	
	iter = find( memoryList.begin(), memoryList.end(), rememberedObject );
	
	// if found, remove process from memory
	if ( iter != memoryList.end() ) {
		
		//trace 
		getTrace() -> mark( this, markForget, dynamic_cast<TraceProducer*>(rememberedObject) );
	
		// observer
		_obsForEach( MemoObserver, Forget( this, rememberedObject ) );
		
		memoryList.erase(iter);
	}
	else {
		
		// error: object not found
		error( "Memo::forget(): object not in memoryList" );
		return false;
	}
	
	return true;
}


void Memo::eraseMemory() {
	
	// remove all registered objects from memory
	memoryList.erase( memoryList.begin(), memoryList.end() );
}


// wake up all CREATED or IDLE processes in memory (i.e. reschedule them)
void Memo::alert() {
	
	if ( memoryList.empty() ) {

		// warning: no object to wake
		warning( "Memo::alert(): no process saved" );
		return;
	}

	// trace
	traceAlert();

	// observer
	_obsForEach( MemoObserver, Alert( this ) );
		
	// iterate over memoryList and schedule processes
	Process* p;
	std::list< Sched* >::iterator iter;
	
	for ( iter = memoryList.begin(); iter != memoryList.end(); ++iter ) {
	
		p = 0;

		// check if next really is a sleeping process
		if ( dynamic_cast<Process*>(*iter) ) {
	
			p = (Process*)(*iter);

			// allow only unscheduled processes to be alerted
			if ( p -> getProcessState() == Process::IDLE ||
				 p -> getProcessState() == Process::CREATED ) {

				// wake up the process
				p -> alertProcess( this );
			}
			else if ( p -> getProcessState() == Process::TERMINATED ) {
			
				// warning: process terminated
				error( "Memo::alert(): process already terminated" );
			}
			else if ( p -> getProcessState() == Process::RUNNABLE ) { 
				
				// warning: process scheduled
				warning( "Memo::alert(): process already scheduled" );
			}
			// no warning for the CURRENT process,
			// events might be running on that stack
		}
		else {
			
			// warning: not a process, no action taken, but continue loop
			warning( "Memo::alert(): remembered object is not a process" ); 
		}
	}
	
	// remove stored objects from memory
	eraseMemory();
}


Memo::MemoType Memo::getMemoType() {

	return type;
}


void Memo::traceAlert() {
	getTrace() -> beginMark( this, markAlert );
	std::list< Sched* >::iterator iter;
	for ( iter = memoryList.begin(); iter != memoryList.end(); ++iter ) {
		getTrace() -> addTag( tagPartner, dynamic_cast<TraceProducer*>(*iter) );
	}
	getTrace() -> endMark();
}


const MarkTypeId Memo::baseMarkId=3000;
const MarkType Memo::markRemember( "remember", baseMarkId+1, typeid(Memo) );
const MarkType Memo::markForget( "forget", baseMarkId+2, typeid(Memo) );
const MarkType Memo::markAlert( "alert", baseMarkId+10, typeid(Memo) );

const TagId Memo::baseTagId = 3000;
const Tag Memo::tagPartner( "partner" );
