//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file TraceFilter.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/06/10

	\brief {brief description}

	\sa TraceFilter.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/TraceFilter.h>
#include <odemx/util/ErrorHandling.h>

#include <cstdio>
#include <cassert>

using namespace odemx;

TraceFilter::TraceFilter() :
	filterOut(false),
	patt("")
{
}

TraceFilter::~TraceFilter() {
}

/**
	TraceFilter provides a filter to remove marks from the trace.
	This filter is set with the string \p f. The string is read
	from left to right. If the keywords \c none or \c all are
	found the filter policy is set. The last occurrence of one of
	these keywords is used. If none of these keywords is found
	\c all is used as pre-set. The \c none policy blocks all marks
	except those specified in the string \p f. The \c
	all policy passes all marks except those specified in the
	string \p f.
	A mark can be specified by its MarkType or its sender. The
	MarkType consists of the MarkTypeId, the MarkTypeName and the
	scope of the mark. Each of these fields can be used to specify
	Marks. The keyword \c mti: is used to specify MarkTypeIds:
	\code
mti:1
mti: 345; mti:1334;
mti: 15, 2354, 2;
	\endcode
	The keyword \c mtn: can be used to specify MarktTypeNames:
	\code
mtn:begin
mtn:end , activate; mtn: destroy;
	\endcode
	The keyword \c mts: is used to specify a scope:
	\code
mts:class odemx::Process
mts:class odemx::Res, class odemx::Bin; mts: class odemx::Waitq;
	\endcode
	The sender can be specified by its label or its type. The
	keyword \c sl: is used to specify the label:
	\code
sl:MyWaitq, Car
sl: Client 2, Client 3; sl: Servant;
	\endcode
	The keyword \c st: specifies the sender type:
	\code
st:class ServantType, class Bin
st:class MyTestProcess1; st:class Storage
	\endcode
	A mark is specified if at least one of the MarkTypeFileds or its
	sender label or its sender type is specified.

	\note The filter can only be set in pause mode or
	before trace starts.
*/
void TraceFilter::setFilter(const char* f) {
	patt = (f!=0 ? f : "");

	// set default and reset
	filter.def=Filter::PASSALL;
	filter.MarkTypeIds.clear();
	filter.MarkTypeNames.clear();
	filter.MarkTypeScopes.clear();
	filter.SenderLabels.clear();
	filter.SenderTypes.clear();
	filterOut=false;

	// scan pattern
	using namespace std;
	string token;
	string::const_iterator i;
	int j;
	enum MODE {ID, MTI, MTN, MTS, SL, ST, DEFALL, DEFNONE, END} parseMode=ID;
	static const string idTab[END]={"", "mti:", "mtn:", "mts:", "sl:", "st:", "all", "none"};

	for (i=patt.begin();; ++i) {
		if (i!=patt.end()) {
			switch(*i) {
			case ' ':
			case '\n':
			case '\t':
				if (parseMode!=ID)
					token+=*i;
				continue;
			case ';':
			case ',':
				break;
			case ':':
				token+=*i;
				if (parseMode==ID)
					break;
				continue;
			default:
				token+=*i;
				continue;
			}
		}

		// remove whitespace at the beginning and at the end
		if (!token.empty() && parseMode!=ID) {
			string::size_type bg = token.find_first_not_of(" \n\t");
			string::size_type eg = token.find_last_not_of(" \n\t");
			token = token.substr(bg, eg+1);
		}

		switch(parseMode) {
		case ID:
			for(j=MTI; j<END && token!=idTab[j]; ++j);
			if (j==DEFNONE) {
				filter.def=Filter::PASSNONE;
				parseMode=ID;
			} else if (j==DEFALL) {
				filter.def=Filter::PASSALL;
				parseMode=ID;
			} else if (j!=END)
				parseMode=MODE(j);
			else {
				error("XMLTrace: error in filter pattern");
				return;
			}
			break;
		case MTI:
			{
				unsigned int l;
				if (sscanf(token.c_str(), "%u", &l)==1)
					filter.MarkTypeIds.insert(l);
				else {
					error("XMLTrace: error in filter pattern");
					return;
				}
			}
			break;
		case MTN:
			filter.MarkTypeNames.insert(token);
			break;
		case MTS:
			filter.MarkTypeScopes.insert(token);
			break;
		case SL:
			filter.SenderLabels.insert(token);
			break;
		case ST:
			filter.SenderTypes.insert(token);
			break;
		default:
			/* MPi unhandled (by Ralf): DEFALL, DEFNONE, END */
			assert(false);
		}

		if (i==patt.end())
			return;

		if (*i==';')
			parseMode=ID;

		token.erase();
	}
}

bool TraceFilter::pass(const MarkType* m, const TraceProducer* sender) {
	std::string str;
	bool match=false;

	if (patt.empty()) {
		filterOut = false;
		return true;
	}

	if (filter.MarkTypeIds.find(m->getId())!=filter.MarkTypeIds.end())
		match=true;

	str=m->getName();
	if (match || filter.MarkTypeNames.find(str)!=filter.MarkTypeNames.end())
		match=true;

	str=m->getScope().name();
	if (match || filter.MarkTypeScopes.find(str)!=filter.MarkTypeScopes.end())
		match=true;

	str=sender->getLabel();
	if (match || filter.SenderLabels.find(str)!=filter.SenderLabels.end())
		match=true;

	str=sender->getType().name();
	if (match || filter.SenderTypes.find(str)!=filter.SenderTypes.end())
		match=true;

	if (filter.def==Filter::PASSALL)
		filterOut=match;
	else if (filter.def==Filter::PASSNONE)
		filterOut=!match;
	else
		filterOut=false;

	return !filterOut;
}
