//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/** \file ExecutionList.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/23

	\brief Declaration of ExecutionList and ExecutionListObserver

	\sa ExecutionList.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_EXECUTIONLIST_INCLUDED
#define ODEMX_EXECUTIONLIST_INCLUDED

#include <odemx/util/SimTime.h>
#include <odemx/util/Observable.h>
#include <odemx/base/Sched.h>

#include <list>


namespace odemx {

	class ExecutionListObserver;

	/** \class ExecutionList

		\ingroup base

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief ExecutionList implements an execution schedule for Sched objects.

		\note ExecutionList supports Observation.

		<!-- [\sa {references to other classes}]* -->

		ExecutionList is used to manage the Sched execution order. A Sched object
		is scheduled at a given time considering its priority and a FIFO- or LIFO-
		strategy, or in relation to an already scheduled object.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class ExecutionList : public Observable< ExecutionListObserver > {
	public:
		/// Construction
		ExecutionList( ExecutionListObserver* o = 0 );
		/// Destruction
		virtual ~ExecutionList();

		/**
			\brief Print the schedule
			
			This function prints a simple representation of the
			execution list to stdout.
		*/
		void printExecutionList();

		/**
			\name Sched Management

			@{
		*/
		/**
			\brief append Sched

			\param p
				pointer to Sched

			Appends Sched \p p at its execution time.
			If there are already Sched objects scheduled at
			the execution time of \p p, addSched will consider
			the priorities and use the FIFO strategy.

			\sa insertSched
		*/
		void addSched(Sched* p);

		/**
			\brief insert Sched

			\param p
				pointer to Sched

			Inserts Sched \p p at its execution time.
			If there are already Sched objects scheduled at
			the execution time of \p p, insertSched will consider
			the priorities and use the LIFO strategy.

			\sa addSched
		*/
		void insertSched(Sched* p);

		/**
			\brief insert Sched \p p after \p previous

			\param p
				pointer to new Sched object

			\param previous
				pointer to Sched object after which the new Sched
				should be inserted

			Inserts Sched \p p after the \p previous Sched object
			in ExecutionList.

			\note insertSchedAfter will change the priority of
			Sched \p p if necessary.

			\sa insertSchedBefore
		*/
		void insertSchedAfter(Sched* p, Sched* previous);

		/**
			\brief insert Sched \p p before \p next

			\param p
				pointer to new Sched

			\param next
				pointer to Sched before which the new Sched
				should be inserted

			Inserts Sched \p p before the \p next Sched
			in ExecutionList.

			\note insertSchedBefore will change the priority of
			Sched \p p if necessary.

			\sa insertSchedAfter
		*/
		void insertSchedBefore(Sched* p, Sched* next);

		/**
			\brief remove Sched

			Removes Sched \p p from ExecutionList.
		*/
		void removeSched(Sched* p);
		//@}

		/**
			\brief top most Sched in ExecutionList

			\return
				pointer to top most Sched in ExecutionList
		*/
		Sched* getNextSched();

		/**
			\brief check if ExecutionList is empty
		*/
		bool isEmpty() {return l.empty();}

		/**
			\brief get simulation time

			\return
				current simulation time
		*/
		virtual SimTime getTime() const = 0;

		// Implementation
	private:
		/// Sched list
		std::list<Sched*> l;

		// allow Sched-derived classes access to private members
		friend class Process;
		friend class Event;

		/**
			\brief inSort Sched \p e in Sched list

			\param e
				pointer to Sched

			\param fifo
				use FIFO or LIFO strategy for inSort
		*/
		void inSort( Sched* e, bool fifo = true );
	};

	/** \interface ExecutionListObserver

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for ExecutionList events.

		<!-- [\note {notes}]* -->

		\sa ExecutionList

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/


	class ExecutionListObserver {
	public:
		virtual ~ExecutionListObserver() {}
		/// construction
		virtual void onCreate(ExecutionList* sender) {}
		/// destruction
		virtual void onDestroy(ExecutionList* sender) {}
		/// Sched added
		virtual void onAddSched(ExecutionList* sender, Sched* newSched) {}
		/// Sched insertion
		virtual void onInsertSched(ExecutionList* sender, Sched* newSched) {}
		/// Sched insertion with parameters
		virtual void onInsertSchedAfter(ExecutionList* sender, Sched* newSched, Sched* previousSched) {}
		/// Sched insertion with parameters
		virtual void onInsertSchedBefore(ExecutionList* sender, Sched* newSched, Sched* nextSched) {}
		/// Sched removed
		virtual void onRemoveSched(ExecutionList* sender, Sched* removedSched) {}
	};
}


#endif

