//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Report.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/06/21

	\brief Declaration of Report mechanism.

	\sa Report.cpp

	The Report mechanism is used to report accumulated data. All Data
	are provided in tables. User defined Report-Objects transform the
	data into some output format (for example XML or HTML).

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_REPORT_INCLUDED
#define ODEMX_REPORT_INCLUDED

#include <odemx/util/SimTime.h>
#include <odemx/util/TypedObject.h>
#include <odemx/util/LabeledObject.h>

#include <sstream>
#include <list>
#include <vector>
#include <string>
#include <typeinfo>

namespace odemx {
	/**
		\ingroup util

		\brief column types

		All columns in tables are associated to a data type.
	*/
	enum ColumnType {
		INTEGER, ///< integer numbers
		REAL, ///< real numbers
		STRING, ///< text
		INVALID ///< used for undefined columns
	};

	/**
		\ingroup util

		\brief control codes

		This control codes are used for operator << input
		into tables.
	*/
	enum CtrlCode {
		ENDL, ///< end of table line
		TAB ///< skip column
	};

	class Report;

	/** \interface TableDefinition

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief The table structure is defined by TableDefinition.

		<!-- [\note {notes}]* -->

		\sa dynTableDefinition utTableDef

		The structure of a table consists of column. Each column has
		a Label and a ColumnType. TableDefinition is an interface that
		is used to provide this information.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class TableDefinition {
	public:
		virtual ~TableDefinition() { }
		/**
			\brief number of columns

			\return number of columns in table
		*/
		virtual unsigned long getNumberOfColumns() const = 0;

		/**
			\brief get label of column

			\return label or 0 if there is no such column
			\param i
				index of column (0..)
		*/
		virtual const char* getLabelOfColumn(unsigned long i) const = 0;

		/**
			\brief get type of column

			\return ColumnType, ColumnType::INVALID if there is no
			such column

			\param i
				index of column (0..)
		*/
		virtual ColumnType getTypeOfColumn(unsigned long i) const = 0;

		friend bool operator ==(const TableDefinition& f, const TableDefinition& s);
	};

	/**
		\ingroup util

		\brief compare two TableDefinitions

		 \return true if number of columns, all labels and all types
		 are equal
	*/
	bool operator ==(const TableDefinition& f, const TableDefinition& s);


	/** \class Table

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Container for reported data.

		<!-- [\note {notes}]* -->

		\sa TableDefinition Report

		All data are reported in tables. The structure of a Table is defined
		by a TableDefinition. A Table cannot be created manually. To get a Table
		the user has to use a Report object.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Table {
		// Interface
	public:
		/// \name Table attributes
		//@{

		/// get Label of table
		const char* getLabel() const {return label.c_str();}

		/// get number of lines in table
		unsigned long getNumberOfLines() const {return numLines;}

		/// get number of columns in table
		unsigned long getNumberOfColumns() const {return def->getNumberOfColumns();}

		/// get column labels
		const char* getLabelOfColumn(unsigned long i) const {return def->getLabelOfColumn(i);}

		/// get column types
		ColumnType getTypeOfColumn(unsigned long i) const {return def->getTypeOfColumn(i);}
		//@}

		/**
			\name Input streaming

			The data can be entered into a table in a streaming like fashion.
			The data type has to match the column type. CtrlCode can be used to skip
			columns or the rest of a line. Skipped columns are filed with default
			data (0, 0.0, "");

			@{
		*/
		friend Table& operator <<(Table& t, int d);
		friend Table& operator <<(Table& t, unsigned int d);
		friend Table& operator <<(Table& t, long d);
		friend Table& operator <<(Table& t, unsigned long d);
		friend Table& operator <<(Table& t, float d);
		friend Table& operator <<(Table& t, double d);
		friend Table& operator <<(Table& t, std::string d);
		friend Table& operator <<(Table& t, const char* d);
		friend Table& operator <<(Table& t, CtrlCode d);
		//@}

		/**
			\name Output streaming

			The data of a table can be read in a streaming like fashion.
			The data type has to match the column type.

			@{
		*/
		friend Table& operator >>(Table& t, long& d);
		friend Table& operator >>(Table& t, double& d);
		friend Table& operator >>(Table& t, std::string& d);
		//@}

		/**
			\name Output random access

			Get the data of a specific field in table.

			@{
		*/
		/**
			\brief Get INTEGER value from field (\p col, \p line).
			\note Column col must be an INTEGER column.
		*/
		long getINTEGER(unsigned long col, unsigned long line);

		/**
			\brief Get REAL value from field (\p col, \p line).
			\note Column col must be an INTEGER or a REAL column.
		*/
		double getREAL(unsigned long col, unsigned long line);

		/**
			\brief Get STRING value from field (\p col, \p line).
			\note Type conversion from INTEGER to STRING and from
			REAL to STRING is done if required.
		*/
		std::string getSTRING(unsigned long col, unsigned long line);
		//@}

		// Implementation
	private:
		// Table description
		std::string label; ///< table label
		TableDefinition* def; ///< table definition

		friend class odemx::Report;
		/// Construction is managed by Report
		Table(const char* l, TableDefinition* d) :
			label(l), def(d),
			outputIntIndex(0), outputRealIndex(0), outputStrIndex(0),
			outputColumnPointer(0), inputColumnPointer(0), numLines(0) {}
		/// Destruction is managed by Report
		~Table() {}

		// Data
		std::vector<long> intData; /// INTEGER data
		std::vector<double> realData; /// REAL data
		std::vector<std::string> stringData; /// STRING data

		/// \name Output streaming control
		//@{
		unsigned long outputIntIndex;
		unsigned long outputRealIndex;
		unsigned long outputStrIndex;

		unsigned long outputColumnPointer;
		//@}

		/// \name Input streaming control
		//@{
		unsigned long inputColumnPointer;
		//@}

		/// size of table
		unsigned long numLines;

		/// \name Help procedures
		//@{
		/// increment column pointer for input streaming, change line if necessary
		void incInpColumnPtr();
		/// increment column pointer for output streaming, change line if necessary
		void incOutpColumnPtr();
		/// add default value (0, 0.0, "") in column
		void addDefaultValue();
		//@}
	};

	Table& operator <<(Table& t, int d); ///< enter int value
	Table& operator <<(Table& t, unsigned int d); ///< enter unsigned int value
	Table& operator <<(Table& t, long d); ///< enter long value
	Table& operator <<(Table& t, unsigned long d); ///< enter unsigned long value
	Table& operator <<(Table& t, float d); ///< enter float value
	Table& operator <<(Table& t, double d); ///< enter double value
	Table& operator <<(Table& t, std::string d); ///< enter std::string value
	Table& operator <<(Table& t, const char* d); ///< enter const char* value
	Table& operator <<(Table& t, CtrlCode d); ///< enter control codes

	Table& operator >>(Table& t, long& d); ///< read long value
	Table& operator >>(Table& t, double& d); ///< read double value
	Table& operator >>(Table& t, std::string& d); ///< read std::string value

	/** \interface ReportProducer

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief A ReportProducer generates data.

		<!-- [\note {notes}]* -->

		\sa Report Table TableDefinition

		A ReportProducer implementation generates data for reports.
		It has to implement virtual void report(Report* r).

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class ReportProducer {
	public:
		virtual ~ReportProducer() { }
		/**
			\brief report generation

			report is called by a Report object during report generation.
			A ReportProducer implementation implements this method to
			provide its data. ReportProducer can be associated to one or
			more Report objects. A ReportProducer will contribute to each
			Report it is associated to.
		*/
		virtual void report(Report* r) = 0;
	};

	/** \class Report

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief baseclass for special report objects

		<!-- [\note {notes}]* -->

		\sa Table TableDefinition ReportProducer

		Report is the base class for user defined report objects.
		Report manages Tables, ReportProducer and the report generation.
		Each Report object represents an individual report. There can be
		multiple reports in a simulation.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Report {
	public:
		Report() {} ///< Construction
		virtual ~Report(); ///< Destruction

		/// \name Table management
		//@{
		/**
			\brief create a Table

			\param name
				name of new Table
			\param def
				TableDefinition of new Table

			\return new Table

			\note If there is already a Table with an equal name and
			definition it will be returned instead of creating a new Table.
		*/
		Table* createTable(const char* name, TableDefinition* def);

		/**
			\brief create a Table

			\param name
				name of new Table
			\param def
				TableDefinition of new Table

			\return new Table

			\note If there is already a Table with an equal name and
			definition it will be returned instead of creating a new Table.
		*/
		Table* createTable(const std::string& name, TableDefinition* def);

		/**
			\brief find a Table

			\param name
				name of Table
			\param def
				TableDefinition of Table

			\return found Table or 0
		*/
		Table* findTable(TableDefinition* def, const char* name);
		//@}

		/// \name ReportProducer handling
		//@{
		/// add ReportProducer \p rp to this Report
		void addProducer(ReportProducer* rp) {rs.push_back(rp);}

		/// remove ReportProducer \p rp from this Report
		void removeProducer(ReportProducer* rp) {rs.remove(rp);}
		//@}

		/**
			\brief generate report

			Report generation:
			-# call all ReportProducer
			-# call processTables
		*/
		void generateReport();

	protected:
		/// contained Tables
		std::vector<Table*> ts;

		/**
			\brief process Tables in this Report

			Report objects have to provide a meaningful processTables() function.
		*/
		virtual void processTables() = 0;

		// Implementation
	private:
		/// associated ReportProducer
		std::list<ReportProducer*> rs;
	};

	/** \class dynTableDefinition

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief dynTableDefinition a TableDefinition implementation for
		dynamic definitions.

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		dynTableDefinition can be used as a TableDefinition in Report::createTable or
		Report::findTable. It can be used to build a TableDefinition at runtime.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class dynTableDefinition : public virtual TableDefinition {
		// Interface
	public:
		virtual ~dynTableDefinition() {} ///< Destruction

		/**
			\name Implementation of TableDefinition interface
			\sa TableDefinition
			@{
		*/
		virtual unsigned long getNumberOfColumns() const {return (unsigned long)columnLabels.size();}
		virtual const char* getLabelOfColumn(unsigned long i) const {if (i>columnLabels.size()) return 0; return columnLabels[i].c_str();}
		virtual ColumnType getTypeOfColumn(unsigned long i) const {if (i>columnTypes.size()) return INVALID; return columnTypes[i];}
		//@}

		/**
			\brief add columns
			\param label
				column label
			\param type
				column type

			use addColumn() to build a TableDefinition
		*/
		void addColumn(const char* label, ColumnType type);

		// Implementation
	private:
		std::vector<std::string> columnLabels; ///< column labels
		std::vector<ColumnType> columnTypes; ///< column types
	};

	/** \class utTableDef

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief utTableDef simplifies report table definitions

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		utTableDef can be used as a TableDefinition in Report::createTable or
		Report::findTable. It can be used to create a TableDefinition without definig a
		new class.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class utTableDef : public virtual TableDefinition {
		unsigned int sz;
		const char** columnLabels;
		const ColumnType* columnTypes;
	public:
		utTableDef(unsigned int size, const char* labels[], const ColumnType types[]);
		virtual ~utTableDef() { }

		/**
			\name Implementation of TableDefinition interface
			\sa TableDefinition
			@{
		*/
		virtual unsigned long getNumberOfColumns() const;
		virtual const char* getLabelOfColumn(unsigned long i) const;
		virtual ColumnType getTypeOfColumn(unsigned long i) const;
		//@}
	};
}

#endif

