//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Report.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/06/21

	\brief Implementation of report mechanism

	\sa Report.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifdef _MSC_VER
#pragma warning(disable : 4786)
#pragma warning(disable : 4530)
#endif

#include <odemx/util/Report.h>
#include <odemx/util/ErrorHandling.h>

#include <cassert>
#include <cstring>
#include <algorithm>

using namespace odemx;
using namespace std;

bool odemx::operator ==(const TableDefinition& f, const TableDefinition& s) {
	unsigned long noCols=f.getNumberOfColumns();
	string strF, strS;

	if (noCols!=s.getNumberOfColumns())
		return false;

	for (unsigned long i=0; i<noCols; ++i) {
		if (f.getTypeOfColumn(i)!=s.getTypeOfColumn(i))
			return false;

		strF=f.getLabelOfColumn(i);
		strS=s.getLabelOfColumn(i);
		if (strF!=strS)
			return false;
	}

	return true;
}

// Input streaming
Table& odemx::operator <<(Table& t, int i) {
	return t << long(i);
}

Table& odemx::operator <<(Table& t, unsigned int i) {
	return t << long(i);
}

Table& odemx::operator <<(Table& t, long d) {
	// type check
	if (t.def->getTypeOfColumn(t.inputColumnPointer)==REAL)
		return t << float(d);

	if (t.def->getTypeOfColumn(t.inputColumnPointer)!=INTEGER) {
		// Error:
		return t;
	}

	t.intData.push_back(d);

	// update inputColumnPointer
	t.incInpColumnPtr();

	return t;
}

Table& odemx::operator <<(Table& t, unsigned long i) {
	return t << long(i);
}

Table& odemx::operator <<(Table& t, float f) {
	return t << double(f);
}

Table& odemx::operator <<(Table& t, double d) {
	// type check
	if (t.def->getTypeOfColumn(t.inputColumnPointer)==INTEGER)
		return t << long(d);

	if (t.def->getTypeOfColumn(t.inputColumnPointer)!=REAL) {
		// Error:
		return t;
	}

	t.realData.push_back(d);

	// update inputColumnPointer
	t.incInpColumnPtr();

	return t;
}

Table& odemx::operator <<(Table& t, string d) {
	return t << d.c_str();
}

Table& odemx::operator <<(Table& t, const char* d) {
	// type check
	if (t.def->getTypeOfColumn(t.inputColumnPointer)!=STRING) {
		// Error:
		return t;
	}

	t.stringData.push_back(d);

	// update inputColumnPointer
	t.incInpColumnPtr();

	return t;
}

Table& odemx::operator <<(Table& t, CtrlCode d) {
	unsigned long r;

	switch (d) {
	case ENDL:
		// fill line with default values
		r=t.numLines;

		while (r==t.numLines)
			// enter default value
			t.addDefaultValue();

		break;
	case TAB:
		// enter default value
		t.addDefaultValue();

		break;
	default:
		// Error:
		return t;
	}

	return t;
}

// Output streaming
Table& odemx::operator >>(Table& t, long& d) {
	// requires ColumnType INTEGER

	// type check
	if (t.def->getTypeOfColumn(t.outputColumnPointer)!=INTEGER) {
		// Error:
		return t;
	}

	// size check
	if (t.outputIntIndex>=t.intData.size())
		t.outputIntIndex=0;

	// write value
	d=t.intData[t.outputIntIndex++];

	// update outputColumnPointer
	t.incOutpColumnPtr();

	return t;
}

Table& odemx::operator >>(Table& t, double& d) {
	// requires ColumnType REAL or INTEGER
	switch (t.def->getTypeOfColumn(t.outputColumnPointer)) {
	case INTEGER:
		// size check
		if (t.outputIntIndex>=t.intData.size())
			t.outputIntIndex=0;

		// write value
		d=double(t.intData[t.outputIntIndex++]);

		break;
	case REAL:
		// size check
		if (t.outputRealIndex>=t.realData.size())
			t.outputRealIndex=0;

		// write value
		d=t.realData[t.outputRealIndex++];

		break;
	default:
		// Error:
		return t;
	}

	// update outputColumnPointer
	t.incOutpColumnPtr();

	return t;
}

Table& odemx::operator >>(Table& t, string& d) {
	// any ColumnType accepted
	ostringstream convert;

	switch (t.def->getTypeOfColumn(t.outputColumnPointer)) {
	case INTEGER:
		// size check
		if (t.outputIntIndex>=t.intData.size())
			t.outputIntIndex=0;

		// convert value
		convert << t.intData[t.outputIntIndex++] << '\0';

		// write value
		d=convert.str();

		break;
	case REAL:
		// size check
		if (t.outputRealIndex>=t.realData.size())
			t.outputRealIndex=0;

		// convert value
		convert << t.realData[t.outputRealIndex++] << '\0';

		// write value
		d=convert.str();

		break;
	case STRING:
		// size check
		if (t.outputStrIndex>=t.stringData.size())
			t.outputStrIndex=0;

		// write value
		d=t.stringData[t.outputStrIndex++];

		break;

	default:
		// Error:
		return t;
	}

	// update outputColumnPointer
	t.incOutpColumnPtr();

	return t;
}

// Output random access
long Table::getINTEGER(unsigned long col, unsigned long line) {
	// requires ColumnType INTEGER
	// type check
	if (def->getTypeOfColumn(col)!=INTEGER) {
		// Error:
		return 0;
	}

	if (line>numLines) {
		// Error:
		return 0;
	}

	// compute integer output index
	unsigned long intOut=0;
	unsigned long numOfIntPerLine=0;
	unsigned long numOfIntBevorCol=0;
	unsigned long nCols= def->getNumberOfColumns();
	unsigned long i;

	for (i=0; i<nCols; ++i) {
		if (def->getTypeOfColumn(i)==INTEGER) {
			numOfIntPerLine++;

			if (i<col)
				numOfIntBevorCol++;
		}
	}

	intOut=line*numOfIntPerLine + numOfIntBevorCol;

	if (intData.size()<intOut) {
		// Error:
		return 0;
	}

	return intData[intOut];
}

double Table::getREAL(unsigned long col, unsigned long line) {
	unsigned long tpOut=0;
	unsigned long numOfTpPerLine=0;
	unsigned long numOfTpBevorCol=0;
	unsigned long nCols= def->getNumberOfColumns();
	unsigned long i;

	if (line>numLines) {
		// Error:
		return 0.0;
	}

	// requires ColumnType REAL or INTEGER
	switch (def->getTypeOfColumn(col)) {
	case INTEGER:
		// compute output index
		for (i=0; i<nCols; ++i) {
			if (def->getTypeOfColumn(i)==INTEGER) {
				numOfTpPerLine++;

				if (i<col)
					numOfTpBevorCol++;
			}
		}

		tpOut=line*numOfTpPerLine + numOfTpBevorCol;

		if (intData.size()<tpOut) {
			// Error:
			return 0.0;
		}

		return double(intData[tpOut]);

	case REAL:
		// compute output index
		for (i=0; i<nCols; ++i) {
			if (def->getTypeOfColumn(i)==REAL) {
				numOfTpPerLine++;

				if (i<col)
					numOfTpBevorCol++;
			}
		}

		tpOut=line*numOfTpPerLine + numOfTpBevorCol;

		if (realData.size()<tpOut) {
			// Error:
			return 0.0;
		}

		return realData[tpOut];

	default:
		// Error:
		return 0.0;
	}
}

string Table::getSTRING(unsigned long col, unsigned long line) {
	unsigned long tpOut=0;
	unsigned long numOfTpPerLine=0;
	unsigned long numOfTpBevorCol=0;
	unsigned long nCols= def->getNumberOfColumns();
	unsigned long i;
	ostringstream convert;

	if (line>numLines) {
		// Error:
		return convert.str();
	}

	// any ColumnType accepted
	switch (def->getTypeOfColumn(col)) {
	case INTEGER:
		// compute output index
		for (i=0; i<nCols; ++i) {
			if (def->getTypeOfColumn(i)==INTEGER) {
				numOfTpPerLine++;

				if (i<col)
					numOfTpBevorCol++;
			}
		}

		tpOut=line*numOfTpPerLine + numOfTpBevorCol;

		if (intData.size()<tpOut) {
			// Error:
			return convert.str();
		}

		convert << intData[tpOut] << '\0';
		return convert.str();

	case REAL:
		// compute output index
		for (i=0; i<nCols; ++i) {
			if (def->getTypeOfColumn(i)==REAL) {
				numOfTpPerLine++;

				if (i<col)
					numOfTpBevorCol++;
			}
		}

		tpOut=line*numOfTpPerLine + numOfTpBevorCol;

		if (realData.size()<tpOut) {
			// Error:
			return convert.str();
		}

		convert << realData[tpOut] << '\0';
		return convert.str();

	case STRING:
		// compute output index
		for (i=0; i<nCols; ++i) {
			if (def->getTypeOfColumn(i)==STRING) {
				numOfTpPerLine++;

				if (i<col)
					numOfTpBevorCol++;
			}
		}

		tpOut=line*numOfTpPerLine + numOfTpBevorCol;

		if (stringData.size()<tpOut) {
			// Error:
			return convert.str();
		}

		return stringData[tpOut];
	default:
		// Error:
		return "";
	}
}

// Help procedures
void Table::incInpColumnPtr() {
	inputColumnPointer++;
	if (inputColumnPointer>=def->getNumberOfColumns()) {
		inputColumnPointer=inputColumnPointer%def->getNumberOfColumns();
		numLines++;
	}
}

void Table::incOutpColumnPtr() {
	outputColumnPointer++;
	if (outputColumnPointer>=def->getNumberOfColumns()) {
		outputColumnPointer=inputColumnPointer%def->getNumberOfColumns();
	}
}

void Table::addDefaultValue() {
	// enter default value
	switch (def->getTypeOfColumn(inputColumnPointer)) {
	case INTEGER:
		intData.push_back(0);
		break;
	case REAL:
		realData.push_back(0.0);
		break;
	case STRING:
		stringData.push_back("");
		break;
	default:
		/* MPi no text for this from Ralf */
		assert(false);
	}

	// update inputColumnPointer
	incInpColumnPtr();
}

Report::~Report() {
	std::vector<Table*>::iterator i;

	for (i=ts.begin(); i!=ts.end(); ++i) {
		delete *i;
	}
}

Table* Report::createTable(const char* name, TableDefinition* def) {
	Table* t=0;

	t=findTable(def, name);
	if (t!=0)
		return t;

	t=new Table(name, def);
	ts.push_back(t);
	return t;
}

Table* Report::createTable(const std::string& name, TableDefinition* def) {
	return createTable(name.c_str(), def);
}

Table* Report::findTable(TableDefinition* def, const char* name) {
	std::vector<Table*>::iterator i;
	std::string str1, str2;

	if (def==0 || name==0)
		return 0;

	for (i=ts.begin(); i!=ts.end(); ++i) {
		if (!(*((*i)->def)==*def))
			continue;

		str1=(*i)->getLabel();
		str2=name;

		if (str1==str2)
			return *i;
	}

	return 0;
}

void Report::generateReport() {
	std::list<ReportProducer*>::iterator i;

	for (i=rs.begin(); i!=rs.end(); ++i)
		(*i)->report(this);

	processTables();
}

void dynTableDefinition::addColumn(const char* label, ColumnType type) {
	if (label==0 || type==INVALID)
		return;

	columnLabels.push_back(label);
	columnTypes.push_back(type);
}

utTableDef::utTableDef(unsigned int size, const char* labels[], const ColumnType types[])
 : sz(size), columnLabels(labels), columnTypes(types)
{
}

unsigned long utTableDef::getNumberOfColumns() const {
	return sz;
}

const char* utTableDef::getLabelOfColumn(unsigned long i) const {
	if (i>=sz)
		return 0;

	return columnLabels[i];
}

ColumnType utTableDef::getTypeOfColumn(unsigned long i) const {
	if (i>=sz)
		return INVALID;

	return columnTypes[i];
}

