#include <odemx/Debug.h>

using namespace odemx;
using namespace std;

/********************* PortHead Template Implementation ***********************/

template < typename ElementType >
PortHeadT< ElementType >::PortHeadT( Simulation* sim, Label portName, PortMode m, 
					unsigned int max, PortHeadObserverT< ElementType >* pho ): 
	Memo( sim, Memo::PORTHEAD ),
	Observable< PortHeadObserverT< ElementType > >( pho ),
	mode( m ),
	env( sim )
	{
		// set label in context
		DefLabeledObject::setLabel( sim, (string(portName) + " (head)").c_str() );
	
		// trace
		getTrace() -> mark( this, markCreate );
		
		// observer
		_obsForEachT(PortHeadObserverT< ElementType >, Create(this));
		
		// create internal port for this head but w/o tail
		if ( max != 0 )
			internalPort = new PortT< ElementType >( env, portName, max, this, 0 );
	}

template < typename ElementType >
PortHeadT< ElementType >::~PortHeadT() 
	{
		// trace
		getTrace() -> mark( this, markDestroy );
		
		// observer
		_obsForEachT(PortHeadObserverT< ElementType >, Destroy(this));		
	
		// delete the PortHead and also the Port
		// if it is not also pointed to by a PortTail
		if ( internalPort && internalPort -> portTail )
			internalPort -> portHead = 0;
		else
			delete internalPort;
	}

template < typename ElementType >
ElementType PortHeadT< ElementType >::get() {
	
	// trace
	getTrace() -> mark( this, markGet, getPartner() );
	
	// observer
	_obsForEachT(PortHeadObserverT< ElementType >, Get(this));		

	while ( internalPort -> elementList.size() == 0 ) {
		
		Sched* caller = env -> getCurrentSched();
		
		switch ( mode ) {
		case WAITING_MODE:

			if ( caller -> getSchedType() == Sched::EVENT ) {
				// error: cannot send an event to sleep
				error( "PortHead::get(): waiting mode not allowed for events" );
				return 0;
			}
			else if ( caller -> getSchedType() == Sched::PROCESS ) {
				Memo::remember( dynamic_cast<Process*>(caller) );
				dynamic_cast<Process*>(caller) -> sleep();
			}
			break;

		case ERROR_MODE:
			error( "PortHead::get(): called on empty port" );
			return 0;

		case ZERO_MODE:
			return 0;
		}
	}
	
	// the Port was full
	bool wasFull = ( internalPort -> elementList.size() == internalPort -> maxCapacity );

	// remove first element from Port
	ElementType first = internalPort -> elementList.front();
	internalPort -> elementList.pop_front();
	
	// update stats
	internalPort -> getCalls++;
	internalPort -> portAccum.update( internalPort -> getNowCount() );
	
	// alert waiting sender
	if ( wasFull && internalPort -> portTail )
		internalPort -> portTail -> alert();

	return first;
}


// insert PortElement at the beginning of the internal Port queue
template < typename ElementType >
bool PortHeadT< ElementType >::unget( ElementType newElement ) {

	// trace
	getTrace() -> mark( this, markUnget, getPartner() );
	
	// observer
	_obsForEachT(PortHeadObserverT< ElementType >, Unget(this, newElement));		

	if ( internalPort -> elementList.size() >= internalPort -> maxCapacity )
		switch ( mode ) {
		case WAITING_MODE:
		case ERROR_MODE:
			error( "PortHead::unget(): called on full port" );
			/* FALLTROUGH */
		case ZERO_MODE:
		return false;
	}
	
	// insert PortElement at the front of the Port queue
	internalPort -> elementList.push_front( newElement );

	// update stats
	internalPort -> ungetCalls++;
	internalPort -> portAccum.update( internalPort -> getNowCount() );

	return true;
}


template < typename ElementType >
PortTailT< ElementType >* PortHeadT< ElementType >::getTail() {

	PortTailT< ElementType >* tail = internalPort -> portTail;

	// tail still zero?
	if ( !tail ) {
		
		// create new Tail
		tail = new PortTailT< ElementType >( env, internalPort -> getLabel(), mode, 0 );
		
		// register new Tail with Head's Port
		internalPort -> portTail = tail;
		
		// register Head's internal Port with the Tail
		tail -> internalPort = this -> internalPort;
	}
	
	return tail;
}


// change
//      old_Port_head -- old_Port -- old_Port_tail
// into
//      old_Port_head -- new_Port (___get new tail___)
//      new_Port_head -- old_Port -- old_Port_tail
// where
//      old_Port_head == this
// return
//      new_Port_head.
template < typename ElementType >
PortHeadT< ElementType >* PortHeadT< ElementType >::cut() {

	// trace
	getTrace() -> mark( this, markCut, getPartner() );
	
	// observer
	_obsForEachT(PortHeadObserverT< ElementType >, Cut(this));		

	// remember old Port
	PortT< ElementType >*     oldPort = internalPort;

	// create a new Head and thereby a new Port
	PortHeadT< ElementType >* newHead = new PortHeadT< ElementType >(env, (string("Cut ") + internalPort -> getLabel()).c_str(), mode, oldPort -> maxCapacity);

	// get the new Port
	PortT< ElementType >*     newPort = newHead -> internalPort;

	// put the new head on the old Port
	oldPort -> portHead  = newHead;
	newHead -> internalPort = oldPort;

	// put the old head (== this) on the new Port
	internalPort = newPort;
	newPort -> portHead = this;

	return newHead;
}


// this PortHead is supposed to be upstream to the portTail oldTail
//   add the contents of this Port to oldTail's Port
//   destroy this, oldTail, and oldTail's Port
//   alert the spliced PortHead and portTail if a significant state
//   change happened
template < typename ElementType >
void PortHeadT< ElementType >::splice( PortTailT< ElementType >* oldTail ) {

	// trace
	getTrace() -> mark( this, markSplice, getPartner() );
	
	// observer
	_obsForEachT(PortHeadObserverT< ElementType >, Splice(this, oldTail));		

	// remember ports
	PortT< ElementType >* tailsPort   = oldTail -> internalPort;
	PortT< ElementType >* headsPort   = this -> internalPort;

	// and their element counts
	int tailsCount = tailsPort -> elementList.size();
	int headsCount = headsPort -> elementList.size();

/*	// append tail's buffer to head's buffer via std::list.splice()
	if ( tailsCount > 0 ) {
		headsPort -> elementList.splice( headsPort -> elementList.end(), tailsPort -> elementList );			
	}

	// patch tail's head onto this head's Port, this PortHead becomes superfluous
	headsPort -> portHead					= tailsPort -> portHead;
	tailsPort -> portHead -> internalPort	= headsPort;
	tailsPort -> portHead = 0;
	
*/	

	// append head's buffer to tail's buffer via std::list.splice()
	if ( headsCount > 0 ) {
		tailsPort -> elementList.splice( tailsPort -> elementList.end(), headsPort -> elementList );			
	}

	// patch tail's head onto this head's Port, this PortHead becomes superfluous
	tailsPort -> portTail					= headsPort -> portTail;
	headsPort -> portTail -> internalPort	= tailsPort;
	headsPort -> portTail = 0;
	
	// delete the throwaways
	oldTail -> internalPort = 0; // so the tail link it won't be deleted in destructor
	delete oldTail;
	delete this;

/*	// alert the new head if its Port becomes non-empty
	if ( (tailsCount == 0) && (headsCount > 0) )
		headsPort -> portHead -> alert();
*/
	// alert the new head if its Port becomes non-empty
	if ( (tailsCount == 0) && (headsCount > 0) )
		tailsPort -> portHead -> alert();
}


template < typename ElementType >
unsigned int PortHeadT< ElementType >::getMaxCapacity() {
      return internalPort -> maxCapacity; 
}


template < typename ElementType >
void PortHeadT< ElementType >::setMaxCapacity( unsigned int newMaximum ) {
	
	unsigned int oldCapacity = internalPort -> maxCapacity;
	
	// trace
	getTrace() -> mark( this, markChangeMaxCapacity, newMaximum, oldCapacity );
	
	// observer
	_obsAForEachT(PortHeadObserverT< ElementType >, MaxCapacity, oldCapacity, newMaximum);
	
	internalPort -> maxCapacity = newMaximum; 
}


template < typename ElementType >
PortMode PortHeadT< ElementType >::getMode() {
	return mode;
}


template < typename ElementType >
void PortHeadT< ElementType >::setMode( PortMode newMode ) {
	
	PortMode oldMode = mode;
	
	// trace
	getTrace() -> mark( this, markChangeMode, newMode, oldMode );
	
	// observer
	_obsAForEachT(PortHeadObserverT< ElementType >, Mode, oldMode, newMode);

	mode = newMode;
}


// return current number of PortElements
template < typename ElementType >
unsigned int PortHeadT< ElementType >::count() {
	return internalPort -> elementList.size(); 
}


// does PortHead contain a PortElement?
template < typename ElementType >
bool PortHeadT< ElementType >::available () {
	return (count() != 0);
}


template < typename ElementType >
void PortHeadT< ElementType >::alert() {
	
	if ( memoryList.empty() ) {

		// warning: no object to wake
		if ( mode != ZERO_MODE ){
			warning( (std::string("PortHead::alert(): no process saved: ") + getLabel()).c_str() );
		}
		return;
	}

	Sched* rememberedObject = memoryList.front();

	// trace
	getTrace() -> mark( this, markAlert, rememberedObject );

	// observer
	_obsForEachT( PortHeadObserverT< ElementType >, Alert( this, rememberedObject ) );

	// check if rememberedObject really is a sleeping process
	if ( rememberedObject -> getSchedType() == Sched::PROCESS ) {

		Process* p = (Process*)rememberedObject;

		// allow only unscheduled processes to be alerted
		if ( p -> getProcessState() == Process::IDLE ||
			 p -> getProcessState() == Process::CREATED ) {

			// wake up the process
			p -> alertProcess( this );
			
			// update stats
			internalPort -> headAlerts++; 
		}
		else if ( p -> getProcessState() == Process::TERMINATED ) {
		
			// warning: process terminated
			error( "PortHead::alert(): process already terminated" );
		}
		else if ( p -> getProcessState() == Process::RUNNABLE ) { 
			
			env -> printExecutionList();
			// warning: process scheduled
			warning( (string("PortHead::alert(): process already scheduled") + this -> getLabel()).c_str() );
		}
		// no warning for the CURRENT process,
		// events might be running on that stack
	}
	else {
		
		// warning: not a process, no action taken, but continue
		warning( "PortHead::alert(): remembered object is not a process" ); 
	}
	
	Memo::forget( rememberedObject );
}

 
template < typename ElementType >
TraceProducer* PortHeadT< ElementType >::getPartner() {

	// return the current process or event, if one is active
	if ( env -> getCurrentSched() != 0 ) {
		
		return dynamic_cast< TraceProducer* >(env -> getCurrentSched());
	}
	else { // or return the simulation 
		
		return env;
	}
}

template < typename ElementType >
const MarkTypeId PortHeadT< ElementType >::baseMarkId = 3100;

template < typename ElementType >
const MarkType PortHeadT< ElementType >::markCreate( "create", baseMarkId+1, typeid(PortHeadT< ElementType >) );
template < typename ElementType >
const MarkType PortHeadT< ElementType >::markDestroy( "destroy", baseMarkId+2, typeid(PortHeadT< ElementType >) );

template < typename ElementType >
const MarkType PortHeadT< ElementType >::markGet( "get", baseMarkId+3, typeid(PortHeadT< ElementType >) );
template < typename ElementType >
const MarkType PortHeadT< ElementType >::markUnget( "unget", baseMarkId+4, typeid(PortHeadT< ElementType >) );
template < typename ElementType >
const MarkType PortHeadT< ElementType >::markCut( "cut", baseMarkId+5, typeid(PortHeadT< ElementType >) );
template < typename ElementType >
const MarkType PortHeadT< ElementType >::markSplice( "splice", baseMarkId+6, typeid(PortHeadT< ElementType >) );
template < typename ElementType >
const MarkType PortHeadT< ElementType >::markAlert( "alert", baseMarkId+7, typeid(PortHeadT< ElementType >) );

template < typename ElementType >
const MarkType PortHeadT< ElementType >::markChangeMode( "changeMode", baseMarkId+8, typeid(PortHeadT< ElementType >) );
template < typename ElementType >
const MarkType PortHeadT< ElementType >::markChangeMaxCapacity( "changeMaxCapacity", baseMarkId+9, typeid(PortHeadT< ElementType >) );

