//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Condq.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/07/11

	\brief Declaration of odemx::Condq

	\sa Condq.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_CONDQ_INCLUDED
#define ODEMX_CONDQ_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/Observable.h>
#include <odemx/util/Report.h>
#include <odemx/util/StatisticObject.h>
#include <odemx/statistics/Statistics.h>
#include <odemx/synchronization/Queue.h>

namespace odemx  {
	class CondqObserver;

	/** \class Condq

		\ingroup synch

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Condq can be used to let a process wait for an
		arbitrary condition.

		\note Condq supports Observation
		\note Condq supports Trace
		\note Condq supports Report

		<!-- [\sa {references to other classes}]* -->

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Condq : public Observable<CondqObserver>,
				  public virtual TraceProducer,
				  public virtual ReportProducer,
				  public DefLabeledObject,
				  public StatisticObject
	{
	public:
		/**
			\brief Construction
			\param s
				pointer to Simulation object
			\param l
				label of this object
			\param o
				initial observer
		*/
		Condq(Simulation* s, Label l, CondqObserver* o = 0);
		/// Destruction
		~Condq();

		/**
			\name Synchronisation
			@{
		*/
		/**
			\brief Wait for cond

			If the supplied condition \p cond is false the current process
			is blocked until the condition is true and signal() was called.
			If a blocked process is interrupted, it is reactivated and
			the function returns false. (An interrupted process has
			influence on the queue-statistic but not on user and waiting time
			statistics.)
		*/
		bool wait(Condition cond);

		/// Trigger condition check
		void signal();
		//@}

		/// Get list of blocked process objects
		const std::list<Process*>& getProcessList() const {return processes.getList();}

		/**
			\name Statistics

			Statistics

			@{
		*/
		/// Reset statistics
		virtual void reset();

		/// Number of processes
		unsigned int getZeroWait() const {return zeros;}

		/// Average slave waiting time
		double getAVWaitTime() const {return sumWaitTime / users;}

		/// Users
		unsigned int getUsers() const {return users;}

		/// Signals
		unsigned int getSignals() const {return signals;}

		//@}

		/// Get pointer to Trace
		virtual Trace* getTrace() const {return env;}

		/**
			\name Condq specific trace marks
			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markWait;
		static const MarkType markContinue;
		static const MarkType markSignal;
		//@}

		/// Generate report
		virtual void report(Report* r);

		// Implementation
	private:
		/// simulation
		Simulation* env;
		/// process management
		Queue processes;

		// statistics
		double sumWaitTime;
		unsigned int users;
		unsigned int zeros;
		unsigned int signals;

		// help methods
		/// current process
		Process* getCurrentProcess() {return env->getCurrentProcess();}
	};

	/** \interface CondqObserver

		\author Ralf Gerstenberger

		\brief Observer for Condq specific events

		<!-- [\note {notes}]* -->

		\sa Condq

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class CondqObserver {
	public:
		virtual ~CondqObserver() { }
		/// Construction
		virtual void onCreate(Condq* sender) {}
		/// Destruction
		virtual void onDestroy(Condq* sender) {}

		/// Process is waiting
		virtual void onWait(Condq* sender, Process* process) {}
		/// Process continues
		virtual void onContinue(Condq* sender, Process* process) {}
		/// Condition test triggered
		virtual void onSignal(Condq* sender, Process* process) {}
	};
}

#endif

