//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file TransmissionEvents.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in TransmissionEvents.h

	\sa TransmissionEvents.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/

#include <odemx/protocol/TransmissionMedium.h>
#include <odemx/protocol/TransmissionEvents.h>
#include <odemx/protocol/ProtocolMessage.h>

using namespace odemx;


UnicastTransmission::UnicastTransmission( ProtocolSimulation* sim, Label l, 
										  ProtocolMessage* m, UnicastTransmissionObserver* o ):
	Event( sim, l ),
	Observable<UnicastTransmissionObserver>( o ),
	message( m )
	{}


void UnicastTransmission::eventAction() {

	TransmissionMedium* medium = 
			((ProtocolSimulation*)getSimulation()) -> getTransmissionMedium();
	
	NetNode* sender = message -> getSendNode();
	NetNode* receiver = message -> getReceiveNode();
		
	// get random sample for connectivity model
	if ( medium -> badConnectivity( sender, receiver ) ) {
					
		reportFailure( message, sender, receiver );
	}
	else {
		
		reportSuccess( message, sender, receiver );
		
		receiver -> receiveFromNet( message );
	}
}



void UnicastTransmission::reportFailure( ProtocolMessage* m,
										 NetNode* sender,
										 NetNode* receiver ) {

	TransmissionMedium* medium = 
		((ProtocolSimulation*)getSimulation()) -> getTransmissionMedium();
	
	medium -> failedUnicastCount++;
	
	//trace
	getTrace() -> beginMark( this, markUnicastFailure );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(sender) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(receiver) );
	getTrace() -> endMark( m -> toString().c_str() ); // msg handed over as comment

	// observer
	_obsForEach( UnicastTransmissionObserver, Failure( this, m, sender, receiver ) );
}



void UnicastTransmission::reportSuccess( ProtocolMessage* m,
										 NetNode* sender,
										 NetNode* receiver ) {

	TransmissionMedium* medium = 
		((ProtocolSimulation*)getSimulation()) -> getTransmissionMedium();
	
	medium -> successfulUnicastCount++;
	
	//trace
	getTrace() -> beginMark( this, markUnicastSuccess );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(sender) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(receiver) );
	getTrace() -> endMark( m -> toString().c_str() ); // msg handed over as comment

	// observer
	_obsForEach( UnicastTransmissionObserver, Success( this, m, sender, receiver ) );
}





BroadcastTransmission::BroadcastTransmission( ProtocolSimulation* sim, Label l, 
											  ProtocolMessage* m, BroadcastTransmissionObserver* o ):
	Event( sim, l ),
	Observable<BroadcastTransmissionObserver>( o ),
	message( m )
	{}


void BroadcastTransmission::eventAction() {
	
	// monitor previous transmissions of the same BROADCAST, copy route if true
	bool previousDeliveries = false;

	NetNode* sender = message -> getSendNode();
	NetNode* receiver = 0;
	
	TransmissionMedium* medium = 
		((ProtocolSimulation*)getSimulation()) -> getTransmissionMedium();
	
	NodeVector& allNodes = medium -> getAllNodes();
	
	int end = allNodes.size();
	for ( int i = 0; i != end; ++i ) {

		receiver = allNodes[i];
		
		if ( medium -> connected( sender, receiver ) ) {
			
			// if current connection to node, transmit the message
			if ( medium -> badConnectivity( sender, receiver ) ) {

				reportFailure( message, sender, receiver );
			}
			else {
				
				reportSuccess( message, sender, receiver );
				
				// make sure duplicate BROADCASTS do not access 
				// the same member data via pointer  
				if ( !previousDeliveries ) {
	
					receiver -> receiveFromNet( message );
					previousDeliveries = true;
				}
				else { // message was broadcast before
	
					ProtocolMessage* messageCopy = message -> deepCopy();
					
					receiver -> receiveFromNet( messageCopy );
				}
			}
		}
	}
}


void BroadcastTransmission::reportFailure( ProtocolMessage* m,
										   NetNode* sender,
										   NetNode* receiver ) {

	TransmissionMedium* medium = 
		((ProtocolSimulation*)getSimulation()) -> getTransmissionMedium();
	
	medium -> failedBroadcastCount++;
	
	//trace
	getTrace() -> beginMark( this, markBroadcastFailure );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(sender) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(receiver) );
	getTrace() -> endMark( m -> toString().c_str() ); // msg handed over as comment

	// observer
	_obsForEach( BroadcastTransmissionObserver, Failure( this, m, sender, receiver ) );
}


void BroadcastTransmission::reportSuccess( ProtocolMessage* m,
										   NetNode* sender,
										   NetNode* receiver ) {

	TransmissionMedium* medium = 
		((ProtocolSimulation*)getSimulation()) -> getTransmissionMedium();
	
	medium -> successfulBroadcastCount++;
	
	//trace
	getTrace() -> beginMark( this, markBroadcastSuccess );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(sender) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(receiver) );
	getTrace() -> endMark( m -> toString().c_str() ); // msg handed over as comment

	// observer
	_obsForEach( BroadcastTransmissionObserver, Success( this, m, sender, receiver ) );
}





const MarkTypeId UnicastTransmission::baseMarkId = 5250;

const MarkType UnicastTransmission::markUnicastSuccess( "unicast success", baseMarkId+5, typeid(UnicastTransmission) );
const MarkType UnicastTransmission::markUnicastFailure( "unicast failure", baseMarkId+5, typeid(UnicastTransmission) );

const TagId UnicastTransmission::baseTagId = 5250;

const Tag UnicastTransmission::tagFrom( "from" );
const Tag UnicastTransmission::tagTo( "to" );


const MarkTypeId BroadcastTransmission::baseMarkId = 5300;

const MarkType BroadcastTransmission::markBroadcastSuccess( "broadcast success", baseMarkId+4, typeid(BroadcastTransmission) );
const MarkType BroadcastTransmission::markBroadcastFailure( "broadcast failure", baseMarkId+4, typeid(BroadcastTransmission) );

const TagId BroadcastTransmission::baseTagId = 5300;

const Tag BroadcastTransmission::tagFrom( "from" );
const Tag BroadcastTransmission::tagTo( "to" );
