//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file CoroutineContext.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/02/04

	\brief Declaration of CoroutineContext, CoroutineContextObserver
	and DefaultContext.

	\sa CoroutineContext.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_COROUTINECONTEXT_INCLUDED
#define ODEMX_COROUTINECONTEXT_INCLUDED

#include <cassert>
#include <typeinfo>
#include <exception>

#include <odemx/coroutine/System.h>
#include <odemx/util/Observable.h>
#include <odemx/util/LabeledObject.h>

namespace odemx {
	class Coroutine;
	class CoroutineContextObserver;

	/** \class CoroutineContext

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Context for executing Coroutines.

		\note CoroutineContext supports Observation.

		\sa CoroutineContextObserver.

		A CoroutineContext encapsulates the main program. It is required for the use of Coroutine.
		Execution can be switched between Coroutine and CoroutineContext.
		The implementation is stack-based for Linux (based on concepts from Stroustrup "Task-Library",
		AT&T 1991 and Hansen "The C++ -Answer Book", Addison Wesley 1990) or Fiber-based
		for Microsoft Windows 98 and later (based on process implementation in ODEM by Martin von Lwis).

		The stack-based implementation should be portable to any platform using linear, continuous stacks
		(growing bottom up or top down) with a C++ (and exception handling) compatible setjmp()/longjmp()
		implementation.

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class CoroutineContext :
		public virtual LabelScope,
		public DefLabeledObject,
		public Observable<CoroutineContextObserver> {
	public:
		/**
			\brief Construction
			\param label
				label of CoroutineContext
			\param o
				pointer to CoroutineContextObserver
		*/
		CoroutineContext(const char* label="", CoroutineContextObserver* o=0);

		~CoroutineContext(); ///< Destruction

		void switchTo(); ///< continue execution in CoroutineContext

		// Implementation
		friend class Coroutine;

	protected:
		Coroutine* getActiveCoroutine(); ///< \return active Coroutine

	private:
		Coroutine* activeCoroutine; ///< active Coroutine

		/** \brief set active Coroutine
			\param c
				pointer to next active Coroutine

			\return previous active Coroutine
			\note There is no more than one Coroutine active in a context at any time.
		*/
		Coroutine* setActiveCoroutine(Coroutine* c);

		unsigned int numberOfCoroutines; ///< counter of Coroutines in this context

		/**
			\brief register new Coroutine in this context
			\param cr
				pointer to new Coroutine
		*/
		void beginCoroutine(Coroutine* cr);

		/**
			\brief Coroutine in this context has finished
			\param cr
				pointer to finished Coroutine
		*/
		void endCoroutine(Coroutine* cr);

#ifdef HAVE_FIBERS
		/// \name Fiber based implementation
		//@{
		LPVOID fiber; ///< pointer to context Fiber
		static bool initFibers; ///< ConvertThreadToFiber called
		//@}

		/// \name Fiber based implementation
		//@{
		void saveFiber(); ///< save current execution point
		//@}

		friend VOID CALLBACK ExecFiber(PVOID);
		friend void FiberSwitch(LPVOID fiber);
#else
		/// \name stack based implementation
		//@{
		/** \brief begin of runtime stack
			\warning managed by Coroutine::initialize
		*/
		StackAdress base;
		StackAdress last; ///< top of runtime stack
		StackBuffer oldStack; ///< saved runtime stack
		int checksum; ///< stack buffer checksum
		jmp_buf env; ///< setjmp, longjmp data
		//@}

		/// \name stack based implementation
		//@{
		bool saveEnvironment(); ///< save current execution point
		void restoreEnvironment(); ///< restore execution point
		//@}
#endif
	};

	CoroutineContext* getDefaultContext(); ///< get the DefaultContext

#ifndef HAVE_FIBERS
	int calcChecksum(StackAdress begin, unsigned long size); ///< detect stack corruption
#endif

	/** \class CoroutineContextObserver

		\author RalfGerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for CoroutineContext specific events.

		<!-- [\note {notes}]* -->

		\sa CoroutineContext

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class CoroutineContextObserver {
		virtual ~CoroutineContextObserver() { }
	public:
		virtual void onCreate(CoroutineContext* sender) {} ///< creation
		virtual void onDestroy(CoroutineContext* sender) {} ///< destruction

		/// execution switches from Coroutine to CoroutineContext
		virtual void onSwitchTo(CoroutineContext* sender, Coroutine* previousActive) {}

		/// active Coroutine changed (oldActive and newActive might be 0)
		virtual void onChangeActiveCoroutine(CoroutineContext* sender, Coroutine* oldActive, Coroutine* newActive) {}
	};

	/** \class DefaultContext

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Default CoroutineContext for convinience.

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		The DefaultContext is provided for convenience. You can
		use this context if you don't want to define your own.

		The DefaultContext is returned by getDefaultContext().

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class DefaultContext : public CoroutineContext {
	private:
		/**
			\brief Construction

			Use getDefaultContext() to get the DefaultContext.
		*/
		DefaultContext() : CoroutineContext("DefaultContext") {}

		friend CoroutineContext* getDefaultContext();
	};
}

#endif

