//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Simulation.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/02/22

	\brief Declaration of odemx::Simulation, odemx::SimulationObserver and odemx::DefaultSimulation

	\sa Simulation.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_SIMULATION_INCLUDED
#define ODEMX_SIMULATION_INCLUDED

#include <list>

#include <odemx/util/Trace.h>
#include <odemx/util/SimTime.h>
#include <odemx/util/Observable.h>
#include <odemx/util/StatisticObject.h>
#include <odemx/random/Dist.h>
#include <odemx/base/ExecutionList.h>
#include <odemx/coroutine/CoroutineContext.h>

namespace odemx {
	
	class Event;
	class Process;
	class SimulationObserver;

	/** \class Simulation

		\ingroup base

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Simulation is the base class for all user simulations.

		\note %Simulation supports Observation.
		\note %Simulation supports Trace.

		<!-- [\sa {references to other classes}]* -->

		In a user simulation
		the method initSimulation() must be implemented to set-up the
		simulation.	run(), step() or runUntil() can be used to compute the
		simulation.	run() returns after the simulation has finished. step()
		returns after each execution of a scheduled object in the simulation. 
		runUntil() returns
		after simulation time exceeds the given time or the simulation is finished.
		%Simulation also manages lists of all Process objects in the states
		CREATED, RUNNABLE, IDLE and TERMINATED.

		\par Example:
		\include matrioschka.cpp

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Simulation : public ExecutionList, public CoroutineContext,
					   public DistContext, public virtual LabelScope,
					   public Trace, public virtual TraceProducer,
					   public virtual StatisticManager,
					   public Observable<SimulationObserver> {
	public:
		/// Process list identifier
		enum List {
			CREATED, ///< Created processes
			RUNNABLE, ///< Runnable processes
			IDLE, ///< Idle processes
			TERMINATED ///< Terminated processes
		};

		// public Interface
	public:
		/**
			\brief Construction
			\param l
				label of this object
			\param o
				initial observer
		*/
		Simulation(Label l = "Simulation", SimulationObserver* o = 0);
		/// Destruction
		virtual ~Simulation();

		/**
			\name Simulation computing

			The computation of a simulation is started with one of
			the following functions. A simulation is finished when there is
			no Sched object left to be executed or the function exitSimulation() 
			was called.

			@{
		*/
		/// Run simulation until it is finished
		void run();
		/// Execute one process and return
		void step();
		/// Run until the current time is greater than \p t
		void runUntil(SimTime t);

		/**
			\brief Simulation status
			\return
				true if simulation is finished

			This function returns true if the simulation is finished.
		*/
		bool isFinished();

		/**
			\brief Simulation computation status
			\return
				true if simulation is currently executing an Event

			This function returns true if the simulation is executing an 
			Event object. This information is needed for process scheduling 
			so that events can schedule processes and still finish their
			eventAction function.
		*/
		bool isExecutingEvent();

		/**
			\brief Stop simulation

			This function stops the simulation.
		*/
		virtual void exitSimulation();

		//@}

		/**
			\name Process management

			Simulation remembers all process objects. The following
			functions can be used to get the objects in different
			process states.

			\sa Process::ProcessState

			@{
		*/
		/**
			\brief Get currently executed process
			\return
				pointer to executed process object

			This function returns a pointer to the momentarily
			executed process (the process in state CURRENT).
		*/
		Process* getCurrentProcess();

		/**
			\brief Get currently executed Sched object
			\return
				pointer to executed Sched object

			This function returns a pointer to the momentarily
			executed object. That can either be the process in 
			state CURRENT, or the currently running event.
		*/
		Sched* getCurrentSched();

		/**
			\brief Get list of processes in state CREATED
			\return
				list of processes in state CREATED

			This function returns a list of all processes
			in state CREATED.
		*/
		std::list<Process*>& getCreatedProcesses();

		/**
			\brief Get list of processes in state RUNNABLE
			\return
				list of processes in state RUNNABLE

			This function returns a list of all processes
			in state RUNNABLE.
		*/
		std::list<Process*>& getRunnableProcesses();

		/**
			\brief Get list of processes in state IDLE
			\return
				list of processes in state IDLE

			This function returns a list of all processes
			in state IDLE.
		*/
		std::list<Process*>& getIdleProcesses();

		/**
			\brief Get list of processes in state TERMINATED
			\return
				list of processes in state TERMINATED

			This function returns a list of all processes
			in state TERMINATED.
		*/
		std::list<Process*>& getTerminatedProcesses();
		//@}

		/**
			\brief Get current simulation time
			\return
				time now

			This function returns the current simulation time.
			The current time is determined by the current or
			last executed process. If a process is executed its
			execution time becomes the new simulation time.
		*/
		virtual SimTime getTime() const { return now; }

		/**
			\brief Get pointer to Trace
			\return
				pointer to trace manager

			This function returns the pointer to the trace
			manager.
		*/
		virtual Trace* getTrace() const {
				return const_cast<Simulation*>(this);
		}

	protected:
		/**
			\brief Initialisation of a simulation

			Implement this method to initialise a simulation.
			One can create and activate the first processes of
			a simulation in this function. Inside this
			function the scheduling operations does not cause
			an immediate execution of a process. The first process
			is executed after this function is finished.
		*/
		virtual void initSimulation() = 0;

	public:
		/**
			\name Trace MarkTypes

			These MarkTypes are used to trace Simulation events.
			A TraceConsumer can use these constants to identify trace
			events send	by Simulation.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markInit;
		static const MarkType markExitCall;
		static const MarkType markRun;
		static const MarkType markStep;
		static const MarkType markRunUntil;
		static const MarkType markExecProc;
		static const MarkType markExecEvent;
		static const MarkType markCurrProc;
		static const MarkType markTime;
		//@}

		// Implementation
	private:
		/// Simulation modes
		enum Mode {
			UNTILEMPTY, ///< compute until execution list is empty
			STEPPING, ///< return after next Sched execution
			UNTILTIME ///< compute until time
		};

		bool isInitialized; ///< simulation is initialised
		bool isStopped; ///< simulation is stopped
		bool executingEvent; ///< simulation currently running executeEvent
		Mode simMode; ///< simulation modus \sa Simulation::Mode
		SimTime endTime; ///< stop time for UNTILTIME simulation
		SimTime now; ///< current time

		Process* current; ///< current process on stack
		Sched* currentSched; ///< current scheduled object: event or process
		std::list<Process*> created; ///< CREATED processes
		std::list<Process*> runnable; ///< RUNNABLE processes
		std::list<Process*> idle; ///< IDLE processes
		std::list<Process*> terminated; ///< TERMINATED processes

		friend class Process;
		friend class Event;
		
		/** 
			\brief Simulation processing
		
			This function is the core of the simulation management. It handles
			the execution list as well as Process and Event execution. Processes
			are executed by switching to their coroutine, Events are triggered
			by calling the respective \p executeEvent() function on that object.
		*/
		void compSimulation(bool inSimulation = true);

		/**
			\name Process Management
			@{
		*/
		void setCurrentSched( Sched* s );
		void setProcessAsCurrent(Process* p);
		void setProcessAsCreated(Process* p);
		void setProcessAsRunnable(Process* p);
		void setProcessAsIdle(Process* p);
		void setProcessAsTerminated(Process* p);
		//@}

		/**
			\name Help procedures
			@{
		*/
		void init();
		void exec();
		void changeProcessList(Process* p, List l);
		void removeProcessFromLists(Process* p);
		//@}
	};

	/**
		\brief Get the DefaultSimulation
		\return
			pointer to the DefaultSimulation object

		The first time getDefaultSimulation() is called, an
		object of type DefaultSimulation is created and returned.
		All following calls to getDefaultSimulation() also return
		this object.

		\sa DefaultSimulation
	*/
	Simulation* getDefaultSimulation();

	/** \interface SimulationObserver

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for Simulation events

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class SimulationObserver :
			public ExecutionListObserver, public CoroutineContextObserver {
	public:
		/// Construction
		virtual void onCreate(Simulation* sender) {}
		/// Destruction
		virtual void onDestroy(Simulation* sender) {}
		/// Initialisation
		virtual void onInitialization(Simulation* sender) {}
		/// Termination
		virtual void onExitSimulation(Simulation* sender) {}
		/// Run simulation
		virtual void onRun(Simulation* sender) {}
		/// Step through simulation
		virtual void onStep(Simulation* sender) {}
		/// Run simulation until time t
		virtual void onRunUntil(Simulation* sender, SimTime t) {}
		/// Process executed
		virtual void onExecuteProcess(Simulation* sender, Process* p) {}
		/// Event executed
		virtual void onExecuteEvent(Simulation* sender, Event* e) {}
		
		/**
			\brief Current process
			\warning
				\p oldCurrent or \p newCurrent maybe 0
		*/
		virtual void onChangeCurrentProcess(Simulation* sender,
						Process* oldCurrent, Process* newCurrent) {}

		/// Process changed list
		virtual void onChangeProcessList(Simulation* sender,
						Process* p, Simulation::List l) {}

		/// Time change
		virtual void onChangeTime(Simulation* sender,
						SimTime oldTime, SimTime newTime) {}
	};

	/** \class DefaultSimulation

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief A default implementation of Simulation provided for convenience.

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		The %DefaultSimulation is provided for convenience. You can
		use this simulation if you don't want to define your own. Use
		getDefaultSimulation() to get a pointer to a %DefaultSimulation
		object. Instead of \p init(), the \p main function of your simulation 
		program will be required to initialize all necessary objects for 
		the start of the simulation.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class DefaultSimulation : public Simulation {
	private:
		/**
			\brief Construction

			Use getDefaultSimulation() to get an object of %DefaultSimulation.

			\sa getDefaultSimulation()
		*/
		DefaultSimulation() : Simulation("DefaultSimulation") {};
		
		friend Simulation* getDefaultSimulation();

	protected:
		virtual void initSimulation() {};
	};
}

#endif /* ODEMX_SIMULATION_INCLUDED */

