//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProcessQueue.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/25

	\brief Declaration of ProcessQueue, ProcessOrder,
	DefaultOrder and PriorityOrder

	\sa ProcessQueue.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_PROCESSQUEUE_INCLUDED
#define ODEMX_PROCESSQUEUE_INCLUDED

#include <list>

namespace odemx {
	class Process;

	/** \struct ProcessOrder
		\author Ralf Gerstenberger
		\brief ProcessOrder is used for customising the Process
		ordering scheme in ProcessQueue
	*/
	struct ProcessOrder {
		virtual ~ProcessOrder() { }
		virtual bool operator ()(const Process& f, const Process& s) = 0;
	};

	/** \struct DefaultOrder
		\author Ralf Gerstenberger
		\brief DefaultOrder implements ProcessOrder
	*/
	extern struct DefaultOrder : public ProcessOrder {
		virtual ~DefaultOrder() { }
		/// \return \p f < \p s
		virtual bool operator ()(const Process& f, const Process& s);
	} defOrder;

	/** \struct PriorityOrder
		\author Ralf Gerstenberger
		\brief DefaultOrder implements ProcessOrder
	*/
	extern struct PriorityOrder : public ProcessOrder {
		virtual ~PriorityOrder() { }
		/// \return \p f.getPriority() > \p s.getPriority()
		virtual bool operator ()(const Process& f, const Process& s);
	} priOrder;

	/** \class ProcessQueue

		\ingroup base

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief ProcessQueue is used for process synchronisation

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		ProcessQueue is a general tool for storing Process objects. It is used
		by synchronisation objects. Every Process can stay in no more than one
		ProcessQueue at a time (provisional). If the priority of a Process in a
		ProcessQueue is changed the ProcessQueue is updated.
		ProcessQueue supports different order schemes.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class ProcessQueue {
		// Interface
	public:
		ProcessQueue(ProcessOrder* pred = &defOrder);
		virtual ~ProcessQueue();

		// Access methods
		Process* getTop() const;
		const std::list<Process*>& getList() const;

		bool isEmpty() const;

		unsigned int getLength() const {return (unsigned int)l.size();}

		// Manipulator methodes
		virtual void popQueue();
		virtual void remove(Process* p);
		virtual void inSort(Process* p,  bool fifo = true);

		// Implementation
	protected:
		std::list<Process*> l;
		ProcessOrder* order;
	};

	/**
		\brief awakes all Process objects in queue \p q
	*/
	void awakeAll(ProcessQueue* q);

	/**
	 	\brief awake first Process object in queue \p q
	*/
	void awakeFirst(ProcessQueue* q);

	/**
		\brief awake next Process to \p p in queue \p q
	*/
	void awakeNext(ProcessQueue* q, Process* p);
}

#endif

