//----------------------------------------------------------------------------
//	Copyright (C) 2002 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file XmlTrace.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/19

	\brief Implementation of XmlTrace

	\sa XmlTrace.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	<!-- [\since {since when in project}] -->
*/

#include <odemx/util/XmlTrace.h>
#include <odemx/util/ErrorHandling.h>
#include <odemx/util/foreach.h>

#include <fstream>
#include <cassert>
#include <cstring>


#if defined(__GNUC__)

#include <cxxabi.h>

int demangleStatus = -1;
#define _clearTypeName(typeIdName) abi::__cxa_demangle(typeIdName, 0, 0, &demangleStatus)

#else /* different compiler */

#define _clearTypeName(typeIdName) typeIdName

#endif /* __GNUC__ detection */


using namespace std;
using namespace odemx;


static const Tag tagPartner( "partner" );
static const Tag tagPartner1( "first partner" );
static const Tag tagPartner2( "second partner" );


XmlTrace::XmlTrace(Simulation* s, string fileName/*=""*/, unsigned int limit/*=2500*/):
	HtmlGenerator( s ),
	sim(s), 
	manageOut(true), 
	pause(false), 
	active(false), 
	filterOut(false), 
	out(&cout),
	xmlMarkCount(0),
	xmlMarkLimit(limit),
	limitReachTime(-1)
	{
		assert(sim!=0);
	
		if ( fileName.empty() ) {
			// output to stdout
			manageOut = false;
		}
		else {
			xmlFileName = fileName;
			openXmlFile();
		}
	}


XmlTrace::XmlTrace(Simulation* s, std::ostream& os):
	HtmlGenerator( s ),
	sim(s), 
	manageOut(false), 
	pause(false), 
	active(false), 
	filterOut(false), 
	out(&os)
	{
		assert(sim!=0);
	
		if (!(*out)) {
			// Error: invalid ostream
			error("XmlTrace(): invalid ostream, sending output to stdout");
			out = &cout;
		}
	}


XmlTrace::~XmlTrace() 
	{}


/**
	XmlTrace provides a filter to remove marks from the trace.
	This filter is set with the string \p f. The string is read
	from left to right. If the keywords \c none or \c all are
	found the filterpolicy is set. The last occurence of one of
	these keywords is used. If none of these keywords is found
	\c all is used as preset. The \c none policy blocks all marks 
	except thouse specified in the string \p f. The \c
	all policy passes all marks except thouse specified in the
	string \p f.
	A mark can be specified by its MarkType or its sender. The 
	MarkType consists of the MarkTypeId, the MarkTypeName and the
	scope of the mark. Each of these fileds can be used to specifie
	Marks. The keyword \c mti: is used to specifie MarkTypeIds:
	\code
mti:1
mti: 345; mti:1334;
mti: 15, 2354, 2;		
	\endcode
	The keyword \c mtn: can be used to specify MarkTypeNames:
	\code
mtn:begin
mtn:end , activate; mtn: destroy;
	\endcode
	The keyword \c mts: is used to specifie a scope:
	\code
mts:class odemx::Process
mts:class odemx::Res, class odemx::Bin; mts: class odemx::Waitq;
	\endcode
	The sender can be specified by its label or its type. The
	keyword \c sl: is used to specifie the label:
	\code
sl:MyWaitq, Car
sl: Client 2, Client 3; sl: Servant;
	\endcode
	The keyword \c st: specifies the sender type:
	\code
st:class ServantType, class Bin
st:class MyTestProcess1; st:class Storage
	\endcode
	A mark is specified if atleast one of the MarkTypeFileds or its
	sender label or its	sender type is specified.

	\note The filter can only be set in pause mode or
	before trace starts.
*/

//
// Simple trace marks are handled by the following methods.
//
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender)) 
		return;

	beginMarkTree(sender, m);
	endMarkTree(comment);
}

// Marks that include information about producer associations:
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* partner, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	beginMarkTree(sender, m);
	writeValue(partner, tagPartner);
	endMarkTree(comment); 
}

void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;
	
	beginMarkTree(sender, m);
	writeValue(firstPartner, tagPartner1);
	writeValue(secondPartner, tagPartner2);
	endMarkTree(comment);
}

// Marks that include information about producer attributes:
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, char newValue, char oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, short newValue, short oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, int newValue, int oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, long newValue, long oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, float newValue, float oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, double newValue, double oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned char newValue, unsigned char oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned short newValue, unsigned short oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned int newValue, unsigned int oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}
void XmlTrace::mark(const TraceProducer* sender, const MarkType* m, unsigned long newValue, unsigned long oldValue, const char* comment/*= 0*/) {
	// filter
	if (!pass(m, sender))
		return;

	writeNvOv(sender, m, newValue, oldValue, comment);
}

//
// Complex trace marks ar transmitted with the following methodes.
// A complex trace mark can be interpreted as a tree of tag-value pairs.
//
// beginMark indicates the construction of a new mark
void XmlTrace::beginMark(const TraceProducer* sender, const MarkType* m) {
	// filter
	if (!pass(m, sender))
		return;

	// open mark tree
	beginMarkTree(sender, m);
}

// endMarks signals the end of construction
void XmlTrace::endMark( const char* comment /*=0*/) {
	// close mark tree
	endMarkTree(comment);
}

// This Methodes are called to describe the construction of complex marks.
void XmlTrace::beginTag(Tag t) {
	openTag(t);
}

void XmlTrace::endTag(Tag) {
	closeTag();
}

void XmlTrace::addTag(Tag t, bool value) {
	writeValue(value, t);
}

void XmlTrace::addTag(Tag t, char value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, short value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, int value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, long value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, float value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, double value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, const char* value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, const TraceProducer* value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, unsigned char value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, unsigned short value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, unsigned int value) {
		writeValue(value, t);
}

void XmlTrace::addTag(Tag t, unsigned long value) {
		writeValue(value, t);
}

//
// trace control
//
// XmlTrace starts a new trace
void XmlTrace::startTrace() {
	active=true;

	beginTraceTree();
}

// XmlTrace stops the trace
void XmlTrace::stopTrace() {
	active=false;

	endTraceTree();
	
	if ( manageOut ) {
		closeXmlFile();
		outputXmlFileList();
		generateHtml( xmlFileName, translate(getFilter()) );
	}
}

// This call signals a break in trace: A TraceConsumer should not produce output after this call is send,
// but await further marks for upkeeping internal data if required.
void XmlTrace::pauseTrace() {
	pause = true;

	*out << "<mark time=\"" << sim -> getTime() << "\" id=\"0000\"";
	*out << " name=\"pause\" scope=\"odemx::Simulation\">" << endl;
	
	char* typeName = _clearTypeName(sim -> getType().name());
	
	*out << "<sender type=\"" << translate(typeName) << "\">";
	*out << translate(sim->getLabel()) << "</sender>" << endl;
		
	// write details tag
	*out << "<tag name=\"start time\">";
	// write time
	*out << sim->getTime() << "</tag>" << endl;

#if defined(__GNUC__)
	free( typeName ); // malloc'ed by demangling function
#endif
}

// This call signals the end of a break in trace: A TraceConsumer should update its output if required,
// and report following marks.
void XmlTrace::continueTrace() {
	pause = false;

	// write details tag
	*out << "<tag name=\"end time\">";
	// write time
	*out << sim->getTime() << "</tag>" << endl;
	*out << "</mark>" << endl;
}


// help functions
void XmlTrace::openXmlFile() {

	// file counter to be added to each file
	static int fileNumber = 0;
	
	// delete last file data
	resetCurrentFile();
	
	// build file name
	ostringstream file;
	file << xmlFileName << "_" << fileNumber++ << ".xml";
	
	// open new file
	out = new ofstream( file.str().c_str() );
	
	if ( out==0 || !(*out) ) {
		// Error: cannot open file
		error( "XmlTrace::openXmlFile(): cannot open file, sending output to stdout" );
		manageOut = false;
		out = &cout;
	}
	else {
		currentFile.name = file.str();
	}
}


void XmlTrace::closeXmlFile() {

	// delete old file pointer
	reinterpret_cast<ofstream*>(out) -> close();
	delete out;
}


FileList& XmlTrace::getXmlFileList() {
	return xmlFiles;
}


void XmlTrace::outputXmlFileList() {
	
	// build file name
	ostringstream file;
	file << xmlFileName << "_file_list.xml";
	
	// open new file
	ostream* xmlStream = new ofstream( file.str().c_str() );
	
	if ( xmlStream==0 || !(*xmlStream) ) {
		// Error: cannot open file
		error( "XmlTrace::outputXmlFileList(): cannot open file, sending output to stdout" );
		xmlStream = &cout;
	}
	
	*xmlStream << "<?xml version=\"1.0\" encoding=\"utf-8\" ?>" << endl;
	*xmlStream << "<xmlfiles simulation=\"" << translate(sim->getLabel()); 
	*xmlStream << "\">" << endl;
	
	FileList::iterator iter;
	for( iter = xmlFiles.begin(); iter != xmlFiles.end(); ++iter ) {

		*xmlStream << "<file name=\"" << translate((*iter).name) << "\" ";
		*xmlStream << "starttime=\"" << (*iter).startTime << "\" ";
		*xmlStream << "endtime=\"" << (*iter).endTime << "\" />" << endl;
	}

	*xmlStream << "</xmlfiles>" << endl;

	reinterpret_cast<ofstream*>(xmlStream) -> close();
	delete xmlStream;
}


void XmlTrace::resetCurrentFile() {
	currentFile.name = "";
	currentFile.startTime = 0;
	currentFile.endTime = 0;
}


void XmlTrace::beginTraceTree() {

	// write XML-Header	
	*out << "<?xml version=\"1.0\" encoding=\"utf-8\" ?>" << endl;
	// open XmlTrace-Tree
	*out << "<xmltrace simulation=\"" << translate(sim->getLabel()) << "\" time=\"" << sim->getTime() << "\">" << endl;
	// remember start time
	currentFile.startTime = sim -> getTime();
}



void XmlTrace::endTraceTree() {
	
	// close XmlTrace-Tree
	*out << "</xmltrace>" << endl;

	currentFile.endTime = sim -> getTime();
	xmlFiles.push_back( currentFile );
}



void XmlTrace::checkMarkCount() { // only called if manageOut is true
	
	// not too many marks continue with current file
	if ( !limitReached() ){
		return;
	}
	// beyond the mark limit    
	else if ( limitReachTime != -1 ) {
		
		 // same time mark must still be added to current file 
		if ( limitReachTime == sim -> getTime() ) {
			return;
		}
		// time change, close file and open new one
		else {
			
			// finish old file
			endTraceTree();
			closeXmlFile();

			// start next file
			openXmlFile();
			beginTraceTree();
			
			// reset
			limitReachTime = -1;
			xmlMarkCount = 0;
		}
		return;
	}
	// limit reached, but time still 0, remember the current time
	else {
		limitReachTime = sim -> getTime();
	}
}

bool XmlTrace::limitReached() { 
	return xmlMarkCount > xmlMarkLimit; 
}

inline void XmlTrace::openXmlTag(const char* n) {
	if (skipOutput())
		return;

	// open Xml tag
	*out << "<" << n << ">";
}

inline void XmlTrace::closeXmlTag(const char* n) {
	if (skipOutput())
		return;

	// close Xml tag
	*out << "</" << n << ">" << endl;
}

void XmlTrace::openTag(Tag t) {
	if (skipOutput())
		return;

	// open tag
	*out << "<tag ";
	if (t.getName()==Tag::NONAME)
		*out << "id=\"" << t.getId() << "\"";
	else
		*out << "name=\"" << translate(t.getName()) << "\"";
	*out << ">";
}

void XmlTrace::closeTag() {
	if (skipOutput())
		return;

	// close tag
	*out << "</tag>" << endl;
}

inline void XmlTrace::writeValue(bool v, const char* n) {
	if (skipOutput())
		return;

	openXmlTag(n);
	// write value
	*out << (v ? "true" : "false");
	closeXmlTag(n);
}

inline void XmlTrace::writeValue(const TraceProducer* tp, const char* n) {
	if (skipOutput())
		return;

	char* typeName = _clearTypeName(tp->getType().name());
	
/*	openXmlTag(n);
	// write partner typename and label
	*out << "<type>" << translate(typeName) << "</type>" << endl;
	*out << "<label>" << translate(tp->getLabel()) << "</label>" << endl;
	closeXmlTag(n);
*/
	*out << "<" << n;	
	*out << " type=\"" << translate(typeName) << "\">";
	*out << translate(tp->getLabel());
	*out << "</" << n << ">" << endl;
}

inline void XmlTrace::writeValue(bool v, Tag t) {
	if (skipOutput())
		return;

	openTag(t);
	// write value
	*out << (v ? "true" : "false");
	closeTag();
}

inline void XmlTrace::writeValue(const TraceProducer* tp, Tag t) {
	if (skipOutput())
		return;

	char* typeName = _clearTypeName(tp->getType().name());
	
/*	openTag(t);
	// write partner typename and label
	*out << "<type>" << translate(typeName) << "</type>" << endl;
	*out << "<label>" << translate(tp->getLabel()) << "</label>" << endl;
	closeTag();
*/
	
	*out << "<tag ";

	if (t.getName()==Tag::NONAME)
		*out << "id=\"" << t.getId() << "\"";
	else
		*out << "name=\"" << translate(t.getName()) << "\"";

	*out << " type=\"" << translate(typeName) << "\">";
	*out << translate(tp->getLabel());
	
	*out << "</tag>" << endl;

#if defined(__GNUC__)
	free( typeName ); // malloc'ed by demangling function
#endif
}

 
const char* XmlTrace::printState( int val ) {
	switch( val ) {
		case( 0 ): return "CREATED";
		case( 1 ): return "CURRENT";
		case( 2 ): return "RUNNABLE";
		case( 3 ): return "IDLE";
		case( 4 ): return "TERMINATED";
	}

	// never reached
	error( "HtmlTrace::printState(): unknown process state" );
	return "Trace ERROR";
}


inline void XmlTrace::beginMarkTree(const TraceProducer* sender, const MarkType* m) {
	if (skipOutput())
		return;

	// if dumping to file, check size and open new one if needed
	if (manageOut)
		checkMarkCount();

	// count marks for file size limit
	++xmlMarkCount;
	
	// open mark tree
	*out << "<mark time=\"" << sim->getTime() << "\" id=\"";
	if (m->getId()==MarkType::NOID)
		*out << "NOID";
	else
		*out << m->getId();
	*out << "\" name=\"";
	if (m->getName()==MarkType::NONAME)
		*out << "NONAME";
	else
		*out << translate(m->getName());

	char* scope = _clearTypeName(m->getScope().name());
	
	*out << "\" scope=\"" << translate(scope) << "\"";
	*out << ">" << endl;

	// write sender
	writeValue(sender, "sender");

#if defined(__GNUC__)
	free( scope ); // malloc'ed by demangling function
#endif
}

inline void XmlTrace::endMarkTree( const char* comment /*=0*/ ) {
	if (skipOutput())
		return;

	if (comment!=0) {
		// writeValue(comment, "comment");
		
		*out << "<tag name=\"comment\">";
		*out << translate( comment );
		//*out << comment;
		*out << "</tag>" << std::endl;
	}
		
	// close mark tree
	*out << "</mark>" << endl;
}


const char* XmlTrace::translate(const char* s) {
	std::string tmp=s;
	return translate(tmp);
}

const char* XmlTrace::translate(const std::string& s) {
	static std::string ret;
	std::string::const_iterator i;

	// prepare ret
	ret.erase();

	// replace forbidden characters
	for (i=s.begin(); i!=s.end(); ++i) {
		switch (*i) {
		case '<':
			ret+="&lt;";
			break;
		case '>':
			ret+="&gt;";
			break;
		default:
			ret+=*i;
		}
	}
	ret+='\0';

	return ret.c_str();
}
