//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Wait.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/06/06

	\brief Implementation of classes in Wait.h

	\sa Wait.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/synchronization/Wait.h>

using namespace std;
using namespace odemx;


Wait::Wait(Simulation* s, Label l) :
	env(s),
	waitForAll(true)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);
}

Wait::Wait(Simulation* s, Label l, Process* oP1) :
	env(s),
	waitForAll(true)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	Process** ppP=&oP1;
	initObservedList(1, ppP);
	wait();
}

Wait::Wait(Simulation* s, Label l, Process* oP1, Process* oP2, bool all) :
	env(s),
	waitForAll(all)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	Process* ppP[2]={oP1, oP2};
	initObservedList(2, ppP);
	wait();
}

Wait::Wait(Simulation* s, Label l, Process* oP1, Process* oP2, Process* oP3, bool all) :
	env(s),
	waitForAll(all)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	Process* ppP[3]={oP1, oP2, oP3};
	initObservedList(3, ppP);
	wait();
}
	
Wait::Wait(Simulation* s, Label l, int size, Process* oP[], bool all) :
	env(s),
	waitForAll(all)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	initObservedList(size, oP);
	wait();
}

Wait::~Wait() {
	// trace
	getTrace()->mark(this, markDestroy);
}

void Wait::addProcess(Process* p) {
	if (p==0)
		return;

	observed.push_back(p);
}

void Wait::removeProcess(Process* p) {
	if (p==0)
		return;

	observed.remove(p);
}

void Wait::setCondition(bool all/*=true*/) {
	waitForAll = all;
}

void Wait::onDestroy(Process* sender) {
	// Remove sender from observed list
	observed.remove(sender);

	// Kick condition check
	waitP->activate();
}

void Wait::onChangeState(Process* sender, Process::State oldState, Process::State newState) {
	// Kick condition check
	waitP->activate();
}

bool Wait::wait() {
	waitP=env->getCurrentProcess();
	list<Process*>::iterator i;

	// trace
	getTrace()->mark(this, markWait, waitP);

	// First check condition
	if (!checkObserved()) {
		// Add this as observer
		for (i=observed.begin(); i!=observed.end(); ++i)
			(*i)->Observable<ProcessObserver>::addObserver(this);
	} 
	else {
		// trace
		getTrace()->mark(this, markContinue, waitP);

		return true;
	}
		

	// Wait for condition
	while (!checkObserved()) {
		waitP->sleep();

		// Handle interrupt
		if (waitP->isInterrupted()) {
			// Remove this as observer
			for (i=observed.begin(); i!=observed.end(); ++i)
				(*i)->Observable<ProcessObserver>::removeObserver(this);

			// trace
			getTrace()->mark(this, markContinue, waitP);

			return false;
		}
	}

	// Remove this as observer
	for (i=observed.begin(); i!=observed.end(); ++i)
		(*i)->Observable<ProcessObserver>::removeObserver(this);

	// trace
	getTrace()->mark(this, markContinue, waitP);

	return true;
}

void Wait::initObservedList(int size, Process* oP[]) {
	// Add entries from oP in observed
	for (int i=0; i<size; ++i) {
		Process* p=oP[i];
		if (p!=0) addProcess(p);
	}
}

bool Wait::checkObserved() {
	// Check Condition: one(all) observed process TERMINATED
	list<Process*>::iterator i;

	if (waitForAll) {
		for (i=observed.begin(); i!=observed.end(); ++i)
			if ( (*i)->getProcessState()!=Process::TERMINATED ) return false;
	
		return true;
	} else {
		for (i=observed.begin(); i!=observed.end(); ++i)
			if ( (*i)->getProcessState()==Process::TERMINATED ) return true;
	
		return false;
	}
}

const MarkTypeId Wait::baseMarkId = 1000;

const MarkType Wait::markCreate = MarkType("create", baseMarkId+1, typeid(Wait));
const MarkType Wait::markDestroy = MarkType("destroy", baseMarkId+2, typeid(Wait));

const MarkType Wait::markWait = MarkType("wait", baseMarkId+10, typeid(Wait));
const MarkType Wait::markContinue = MarkType("continue", baseMarkId+11, typeid(Wait));

const TagId Wait::baseTagId = 1000;

