//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Rdist.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/09

	\brief Implementation of classes in Rdist.h

	\sa Rdist.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <cmath>
#include <iostream>

#include <odemx/random/Rdist.h>
#include <odemx/util/ErrorHandling.h>

using namespace odemx;

Normal::Normal(DistContext* c, Label title, double na, double nb)
	: Rdist(c, title)
{
	if (nb < 0.0) {
		nb = -nb;
		warning("Normal: nb < 0; -nb used");
	}
	a=na; b=nb; zyqeven=0;
	zyqu=zyqv=0.0;
	uses=0;
}

double Normal::sample() {
	double z;
	uses++;
	if (zyqeven) {
		zyqeven=0;
		z= zyqu*cos(zyqv);
	}
	else {
		zyqeven= 1;
		zyqu= sqrt(-2.0*log(getSample()));
		zyqv= 6.2831853*getSample();
		z= zyqu*sin(zyqv);
	};
	return (z*b + a);
};

void Normal::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Normal" << long(getUses()) << long(getSeed()) << a << b << ENDL;
}

Negexp::Negexp (DistContext* c, Label title, double na) : Rdist(c, title)
{
	if (na <= 0.0) {
		na = (na<0.0) ? -na : 0.001;
		warning("Negexp: na <= 0; -na or 0.001 used");
	}
	a=na;
	uses=0;
};

double Negexp::sample() {
	uses++;
	return( -log(getSample())/a);
}

void Negexp::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Negexp" << long(getUses()) << long(getSeed()) << a << ENDL;
}

Uniform::Uniform (DistContext* c, Label title, double ua, double ub)
 : Rdist(c, title)
{
	if (ua>ub) {
		double help= ua; ua= ub; ub= help;
		warning("Uniform: ua > ub; bounds swapped");
	}
	a= ua; b= ub;
	zyqspan= b-a;
	uses=0;
};

double Uniform::sample() {
	uses++;
	return( zyqspan*getSample() + a);
}

void Uniform::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Uniform" << long(getUses()) << long(getSeed()) << a << b << ENDL;
}

Erlang::Erlang (DistContext* c, Label title, double ea, int eb)
 : Rdist(c, title)
{
	if (ea<=0.0) {
		ea= (ea<0.0) ? -ea : 0.01;
		warning("Erlang: ea<=0; -ea or 0.01 used");
	}
	if (eb<=0) {
		eb= (eb<0) ? -eb : 1;
		warning("Erlang: eb<=0; -eb or 1 used");
	}
	a=ea; b= eb;
	zyqab= a/b;
	uses=0;
};

double Erlang::sample() {
	int k;
	double prod;

	uses++;
	prod = getSample();
	for (k=2; k<=b; k++)  prod*=getSample();
	return ( -log(prod)*zyqab);
};

void Erlang::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Erlang" << long(getUses()) << long(getSeed()) << a << b << ENDL;
}

void Rconst::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Rconst" << long(getUses()) << long(getSeed()) << a << ENDL;
}

