//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProtocolSimulation.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::ProtocolSimulation

	\sa ProtocolSimulation.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#ifndef ODEMX_PROTOCOL_SIMULATION_INCLUDED
#define ODEMX_PROTOCOL_SIMULATION_INCLUDED


#include <odemx/base/Simulation.h>
#include <odemx/protocol/ProtocolLayer.h>


namespace odemx {

	// forward declaration
	class NetTopology;
	class TransmissionMedium;
	class ProtocolLayerManager;

	
	/** \class ProtocolSimulation

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %ProtocolSimulation is the base class for ODEMx protocol simulations
		
		\sa ProtocolLayerManager, NetTopology, TransmissionMedium

		ProtocolSimulation is a specialization of Simulation. It is the context
		for protocol model components and offers several functions to access
		the NetTopology, the TransmissionMedium and the ProtocolLayers.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class ProtocolSimulation: public Simulation,
							  public virtual ReportProducer {
	public:
		
		
		/**
			\brief Construction
			\param l
				label of this object
			\param o
				initial observer
		*/
		ProtocolSimulation( Label l = "ProtocolSimulation",
							SimulationObserver* o = 0 );
		
		/// Destruction
		virtual ~ProtocolSimulation();
		
		/// get access to the network's topology
		NetTopology* getNetTopology();
		
		/// get a pointer to the TransmissionMedium
		TransmissionMedium* getTransmissionMedium();
	
		/// get the simulation's protocol layer manager
		ProtocolLayerManager* getLayerManager();
		
		
		/**
			\brief get a list of all layers

			This function allows external access to the node's input port.
			The given element \p msg will be put into the node's input port.
		*/
		ProtocolLayerList& getLayerList();

		
		/**
			\brief template function for layer creation
			\param l
				Label of this object
			\param o
				initial observer

			This function must be called with a ProtocolEntity type as
			template parameter. A new ProtocolLayerT object will be
			created and automatically registered with the layer manager. 
		*/
		template < typename EntityType >
		void createLayer( Label l, ProtocolLayerObserver* o = 0 ) {
			new ProtocolLayerT< EntityType >( this, l, o );
		}


		/**
			\brief print all created layers
			\param withEntities
				determine whether to show all entities

			This function prints a protocol simulation's layers. If
			\c withEntities is set to \c true, all entities registered 
			with a layer will be displayed, which are as many as there 
			are nodes in the network. 
		*/
		void printLayers( bool withEntities = false );
		
		
		/**
			\brief report generation

			This function is called by a Report object during report generation.
			A ProtocolSimulation uses this method to provide its data. It will 
			contribute to each Report it is associated to.
		*/
		virtual void report( Report* r );

	protected:
		
		
		/**
			\brief Initialisation of a protocol simulation

			Implement this method to initialise a protocol simulation.
			One can create and activate the first processes of
			a simulation in this function.
		*/
		virtual void initSimulation() = 0;
		
		
	private:

		friend class TransmissionMedium;
		
		
		/**
			\brief set the TransmissionMedium
			\param tm
				transmission medium to be used by this simulation

			Set the TransmissionMedium used by this Simulation.
			Low level ProtocolEntities automatically access the medium
			to transmit messages.
		*/
		void setTransmissionMedium( TransmissionMedium* tm );
		
		/// pointer to the TransmissionMedium
		TransmissionMedium* medium;
		/// pointer to the ProtocolLayerManager
		ProtocolLayerManager* layerManager;
	};



}



#endif /* ODEMX_PROTOCOL_SIMULATION_INCLUDED */

