//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file NetNode.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::NetNode

	\sa NetNode.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#ifndef ODEMX_NET_NODE_INCLUDED
#define ODEMX_NET_NODE_INCLUDED

#include <odemx/synchronization/PortTemplate.h>
#include <odemx/protocol/ProtocolEntity.h>
#include <odemx/protocol/ProtocolLayer.h>

#include <map>



namespace odemx {

	// forward declaration
	class TransmissionMedium;
	class ProtocolSimulation;
	class ProtocolMessage;

	class NetNodeObserver;
	
	
	/** \class NetNode

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %NetNode is the base class for nodes in protocol simulations
		
		\note %NetNode supports Observation.
		\note %NetNode supports Trace.

		\sa NetTopology, ProtocolLayer, ProtocolEntity, ProtocolSimulation

		NetNodes are the basic structural components of network models.
		Following the OSI layer model for network protocols, a node manages
		exactly one ProtocolEntity per protocol layer. All ProtocolEntities 
		belonging to a NetNode are automatically created and linked according 
		to their layers upon construction of that node. 

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class NetNode: public Process,
				   public Observable<NetNodeObserver> {

	public:
		
		/**
			\brief Construction
			\param sim
				pointer to the Simulation object
			\param l
				label of this object
		*/
		NetNode( ProtocolSimulation* sim, Label l, NetNodeObserver* o = 0 );
	
		/// Destruction
		virtual ~NetNode();

		
	protected:
		
		/// Node lifecycle to be defined
		virtual int main() = 0;


	public:

		
		/**
			\name Node port access

			These functions provide access to a node's input and output
			ports. The input port can be written, and the output port can 
			be read. Processes can use the respective PortHead or PortTail
			as an alerter when waiting for sending or receiving messages.

			@{
		*/
		/**
			\brief node message input
			\param msg
				message from the node's environment to put into input port

			This function allows external access to the node's input port.
			The given element \p msg will be put into the node's input port.
		*/
		void input( PortElement* msg );
		

		/**
			\brief node message output
			\param msg
				message to be delivered to the node's environment

			This function allows access to the node's output port. The given 
			element \p msg will be put into the node's output port, which
			may then be read by the node's environment. 
		*/
		void output( PortElement* msg );
		
		
		/**
			\brief node message input from the physical layer
			\param msg
				message the node receives from the network

			This function provides access to a node's physical layer entity.
			The given message \p msg will be put into the bottom layer 
			entity's input port.
		*/
		void receiveFromNet( ProtocolMessage* msg );

		
		/**
			\brief transmit a message
			\param e
				message in form of a port element
				
			This function can be used to send a message through the network 
			via the NetNode's top layer entity. The element \p e will be 
			passed on to that entity, which should evaluate it and act 
			according to the element's type.
		*/
		void transmit( PortElement* e );
	
		
		/**
			\brief get a node's output port
			\return
				the head of the node's output port

			This function returns a pointer to the output port head,
			which may be used by other processes as alerter. All messages
			delivered with \p output() can be read from this port head.   
		*/		
		PortHead* outputPort();
		
		
		/**
			\brief get a node's input port
			\return
				the tail of the node's input port

			This function returns a pointer to the input port tail,
			which may be used by other processes as alerter.   
		*/
		PortTail* inputPort();
				
		
		/**
			\brief read node output port
			\return
				the first element from the node's output port 

			This function can be used to read the node's output port.
		*/
		PortElement* readOutputPort();

	protected:
		
		/**
			\brief read node input port
			\return
				first element from the input port
			
			This function provides access to the node's input port.
		*/
		PortElement* readInputPort();

		/**
			\brief get a node's input port
			\return
				the head of the node's input port

			This function returns a pointer to the input port head,
			which can be used by this process as alerter.   
		*/
		PortHead* inputPortHead();
		//@}
		
		/**
			\name Protocol layers and corresponsing entities

			These functions provide access to a node's entities
			which provide different functionality, according
			to the layer of the protocol model.

			@{
		*/		
		/**
			\brief get a list of all layers
			\return
				reference to the simulation's layer list
			
			This function allows access to the simulation's layer list, which
			can then be used to retrieve a pointer to a protocol layer.
		*/
		ProtocolLayerList& getLayerList();
		
		
		/**
			\brief initialize all of the node's entities
			\param layers
				reference to the simulation's layer list
			
			This function initializes and links all entities belonging to
			a node. For each layer in the list, one entity will be created
			and linked to its corresponding lower and upper layer entity.
			The function is called automatically during NetNode construction.
		*/
		virtual void initEntities( ProtocolLayerList& layers );


	private:
		
		friend class ProtocolEntity;
		
		/**
			\brief access the simulation's layer manager
			\return
				the current simulation's layer manager
			
			This function provides access to the protocol layer manager
			which stores all layers of the protocol model.
		*/
		ProtocolLayerManager* getLayerManager();
		
		
		/**
			\brief get the entity representing the top layer
			\return
				top layer entity
		*/
		ProtocolEntity* getTopLayerEntity();

		
		/**
			\brief get the entity representing the bottom layer
			\return
				bottom layer entity
				
			This function returns the lowest layer entity, which
			is connected to the TransmissionMedium.
		*/
		ProtocolEntity* getBottomLayerEntity();
		
		
		/**
			\brief get the entity representing \p layer
			\param layer
				pointer to the layer that contains the requested entity
			\return
				a protocol entity in the specified layer
		*/
		ProtocolEntity* getEntityByLayer( ProtocolLayer* layer );

		
		/**
			\brief get the entity representing layer \p l
			\param l
				label of the layer that contains the requested entity
			\return
				a protocol entity in the specified layer
		*/
		ProtocolEntity* getEntityByLayerName( Label l );
		//@}
		
	private:
	
		PortHead* inPort;	///< user data to send: NetNode uses get()
		PortTail* outPort;	///< receives data for user: NetNode uses put()

		/// associative array linking protocol layer and this node's entities
		std::map< ProtocolLayer*, ProtocolEntity* > entities;
		ProtocolEntity* topEntity;
		ProtocolEntity* bottomEntity;
		
		
	public:
		
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetNode calls.
			A TraceConsumer can use these constants to identify additional 
			trace events (besides those from base class Process) sent by 
			NetNode.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markInput;
		static const MarkType markOutput;
		static const MarkType markTransmit;
		static const MarkType markReceiveFromNet;
		//@}
	};

		
	
	/** \interface NetNodeObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for NetNode specific simulation events.

		<!-- [\note {notes}]* -->

		\sa NetNode, ProcessObserver

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class NetNodeObserver: public ProcessObserver {
		
	public:
		/// Destructor
		virtual ~NetNodeObserver() {}
				
		/// Input from environment
		virtual void onInput( NetNode* sender, PortElement* message ) {}
		/// Output to environment
		virtual void onOutput( NetNode* sender, PortElement* message ) {}
		/// transmit a message
		virtual void onTransmit( NetNode* sender, PortElement* message ) {}
		/// Reception from network
		virtual void onReceiveFromNet( NetNode* sender, ProtocolMessage* message ) {}	
	};

}

#endif /* ODEMX_NET_NODE_INCLUDED */

