//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file SimpleProtocolEntities.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/06/10

	\brief Example for ODEMx protocol simulation techniques

	\sa SimpleProtocolMessage.h SimpleProtocol.cpp

	This example shows essential techniques for  
	implementing a protocol simulation with ODEMx.
	Protocol behavior is specified by employing the OSI
	layer model.

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/

/** 
	\example SimpleProtocolEntities.h

	With this example, the essential protocol simulation techniques of 
	ODEMx are introduced.

	Protocol models for ODEMx are based on the OSI layer model approach,
	where the protocol stack is separated into several layers, each with 
	a distinct function.

	A protocol simulation starts out with a ProtocolSimulation object, 
	which is basically a normal Simulation object, extended by functions
	to create and access the network topology, the transmission medium,
	and the protocol layers. 
	
	The network topology is described by NetNode objects and their
	connections, which are maintained by the class NetTopology. Derived 
	from this class is the TransmissionMedium, which also handles
	message transmissions between the network's nodes.
	
	A node itself has an input and an output port, which can be used 
	by the system environment to pass data to a node, and to register 
	successful message transmissions at a receiving node. The behavior
	of a node is modeled in its active components, so-called entities.
	
	A ProtocolEntity is a process representing a protocol layer function.
	Each node maintains exactly one entity per layer, which are 
	automatically created upon NetNode construction. Entities can receive
	input from the layer above or below and a behavior must be specified
	according to the data passed between these layers.

	In this example, a simple protocol simulation is defined, which shows
	the essential techniques to model and run a protocol simulation with
	ODEMx. Here, two protocol layers are modeled, one for the hardware
	part of a node, and one for the software part. The network contains 
	simple nodes, whose behavior is modeled by the definition of a
	HardwareEntity and a SoftwareEntity. The network's topology is
	definied in the simulation objects's initialization function. Lastly,
	messages between nodes are triggered by external events, which are also
	scheduled during the initialization phase.
	
	\sa SimpleProtocolMessage.h SimpleProtocol.cpp
*/


#ifndef PROTOCOLEXAMPLEENTITIES_H_
#define PROTOCOLEXAMPLEENTITIES_H_

#include <odemx/odemx.h>
#include <set>


using namespace odemx;
using namespace std;

//
// This protocol entity models the behavior of a node's hardware
// layer. It handles transmission and reception of low-level messages.
// Received messages are acknowledged, sent messages require an 
// acknowledging message from the receiver. The wait period for this 
// message is limited by a timer. 
//
class HardwareEntity: public ProtocolEntity {
public:
	HardwareEntity( ProtocolSimulation* sim, ProtocolLayer* layer, NetNode* owner ):
		ProtocolEntity( sim, layer, owner ),
		repeatedUCasts( 0 ),
		waiting( false )
		{
			uCastTimer = new Timer( sim, (string( owner -> getLabel() )+" Unicast Timer").c_str() );
		}
	~HardwareEntity() 
		{
			delete uCastTimer;
		}
	//
	// Definition of this function is required for every protocol
	// entity as it is called by the life cycle function when 
	// the process is awakened by its bottom port.
	//
	// Received messages come from the transmission medium. These 
	// are acknowledged by sending a special message type to the 
	// previous sender, after which the received message is passed 
	// to the software layer.
	//
	virtual void handleLowerLayerInput( PortElement* e ) {
		Message* m = (Message*)e;
		//
		// Store the type of the last message; evaluated when waiting for ACK
		//
		typeOfLastMsg = m -> getMessageType();
		//
		// Check for unidirectional messages
		//
		if ( m -> isUnicast() ) {
			if ( typeOfLastMsg == Message::ACK ) {
				//
				// If an acknowledging message is received, the wait
				// timer is stopped and the repeated sends counter reset.
				// Acknowledging messages are not passed to the next layer.
				//
				if ( uCastTimer -> isScheduled() ) {
					uCastTimer -> stop();
				}
				repeatedUCasts = 0;
				return;
			}
			else {
				//
				// Besides ACK, all unidirectional messages are acknowledged. 
				//
				acknowledgeMessage( m );
			}
		}
		//
		// Pass any received message besides ACK to this node's SoftwareEntity.
		//
		passUpward( m );
	}
	//
	// This function definition is also required by ProtocolEntity.
	// It ensures that messages coming from the layer above are handled
	// properly.
	//
	// Messages coming from the software layer are evaluated by type.
	// The hardware layer sets the sender node, whereby route requests 
	// are transmitted immediately because they are modeled as broadcast
	// messages to all of a node's neighbors. Unidirectional messages
	// require the setting of the next hop, i.e. the receiving node,
	// after which the node will go into a wait state for an acknowledgement
	// of the sent message.
	//
	virtual void handleUpperLayerInput( PortElement* e ) {
		Message* m = (Message*)e;
		//
		// Set sender.
		//
		m -> sendNode = this -> getNode();
		//
		// Broadcast: no receiver handling, no timer. Simply transmit.
		//
		if ( m -> getMessageType() == Message::RREQ ) {
			transmit( m );
			return;
		}
		//
		// Unicast: if the routeMarker points to next node on route,
		// set that route as the next receiver of this message.
		//
		if ( m -> getNextHop() ) {
			m -> receiveNode = m -> getNextHop();
		}
		else {
			error( "HardwareEntity::handleUpperLayerInput(): next hop not set" );
			return;
		}
		//
		// Count the number of transmissions of m, set the time limit for 
		// ACK wait time, and transmit the message via medium.
		//
		repeatedUCasts++;
		uCastTimer -> setIn( m -> getTransmissionDuration() + 5 );
		transmit( m );
		//
		// Avoid wait state recursion.
		//
		if ( waiting ) {
			return;
		}
		waiting = true;
		waitForACK( m );
		waiting = false;
	}
	//
	// Acknowledgement of a message is modeled by returning a shallow 
	// copy of the received message to the sender, setting its type to ACK.
	// This is especially useful for debugging.
	//
	void acknowledgeMessage( Message* m ) {
		Message* ack = new Message( *m );
		ack -> messageType = Message::ACK;
		ack -> sendNode = this -> getNode();
		ack -> receiveNode = m -> getSendNode();
		transmit( ack );
	}
	//
	// This function describes the waiting state for acknowledgements
	// or a timeout. Other messages are forwarded
	//	
	void waitForACK( Message* m ) {
		while( true ) {
			//
			// Wait for an ACK, or a timeout. Handle other messages as usual.
			//
			Memo* theAlerter = wait( uCastTimer, bottomPortHead(), topPortHead() );
			
			if ( theAlerter == bottomPortHead() ) {
				handleLowerLayerInput( readBottom() );
				//
				// The type of the last received message is set in
				// handleLowerLayerInput().
				//
				if ( typeOfLastMsg == Message::ACK ) break;
				else continue;
			}
			else if ( theAlerter == topPortHead() ) {
					
				handleUpperLayerInput( readTop() );
			}
			//
			// ACK timeout has occurred. If still below the retransmission
			// limit, send the message again, otherwise report the broken link. 
			//
			else if ( theAlerter == uCastTimer ){
				
				if ( repeatedUCasts < maxRepeatUCast ) {
					// retransmit message
					repeatedUCasts++;
					uCastTimer -> setIn( 15 );
					transmit( m );
					continue;
				}
				else {
					cout << "Currently no link between " 
						 << m -> getSendNode() -> getLabel()
						 << " and " << m -> getReceiveNode() -> getLabel() 
						 << " for: " << m -> toString() << endl;
					return;
				}
				break;
			}
			else if ( theAlerter == 0 && isInterrupted() )
				error( "HardwareEntity::waitForACK(): wait phase interrupted" );
			else
				error( "HardwareEntity::waitForACK(): cannot determine alerter" );
		}
	}
private:
	Message::Type typeOfLastMsg;
	int repeatedUCasts;
	Timer* uCastTimer;
	bool waiting;
	static const int maxRepeatUCast; 
};

//
// This protocol entity models the behavior of a node's software
// layer. It handles the routing table, which only contains complete 
// routes from previously established connections. When a message is 
// to be broadcast, the route to the destination is determined on demand.
//
class SoftwareEntity: public ProtocolEntity {
public:	
	bool waiting;
	Timer* RREQTimer;
	RouteTable routeTable;
	set< int > seenRREQs;
	set< int > expectedRREPs;
	
	static int messageCount;
	static const int maxRepeatRREQ;

	SoftwareEntity( ProtocolSimulation* sim, ProtocolLayer* layer, NetNode* owner ):
		ProtocolEntity( sim, layer, owner ),
		waiting( false )
		{
			RREQTimer = new Timer( sim, (string(owner->getLabel())+" RREQ Timer").c_str() );
		}

	~SoftwareEntity()
		{}
	//
	// Definition of this function is required by ProtocolEntity.
	// The software layer handles user data input. The UserData
	// object contains the source and destination nodes as well
	// as the actual message data.
	//
	virtual void handleUpperLayerInput( PortElement* e ) {
		UserData* u = (UserData*)e;
		//
		// Create a new protocol message object which will remain
		// the same during the transmissions. Only when it is broadcast,
		// copies are made.
		//
		Message* m = new Message( Message::UDAT, messageCount++, u );
		//
		// Check for an existing route. If one is found, send a data
		// packet. Otherwise, broadcast a route request.
		//
		Route* searchRoute = lookupRoute( m -> getDestination() );

		if ( searchRoute != 0 ) {
			m -> messageType = Message::DATA;
			m -> route = searchRoute;
			m -> routeMarker = 0;
			++( m -> routeMarker );
			transmit( m );
		}
		else {
			sendRREQ( m );
			//
			// Avoid wait state recursion.
			//
			if ( waiting )
				return;
			
			waiting = true;
			waitForRREP( m );
			waiting = false;
		}
	}
	//
	// Definition of this function is required by ProtocolEntity.
	// The software layer evaluates input from the HardwareEntity
	// and handles the message according to its type.
	//
	virtual void handleLowerLayerInput( PortElement* e ) {

		Message* m = (Message*)e;
		
		switch ( m -> getMessageType() ) {
		//
		// DATA packets for this node are passed to the node's output port.
		// Otherwise they are simply forwarded to the next hop.
		//
		case Message::DATA:

			if ( this -> getNode() == m -> getDestination() ) {
				passUpward( m -> userData );
			}
			else {
				++( m -> routeMarker );
				transmit( m );
			}
		break;
		//
		// Route requests are stored as previously seen so that multiple
		// receptions of the same message are ignored. An RREQ for this 
		// node is anwered with a route reply. Old RREQs are ignored,
		// new RREQs for other nodes are forwarded.
		//
		case Message::RREQ:
			//
			// This node is the destination of the RREQ
			//
			if ( this -> getNode() == m -> getDestination() ) { 
				m -> addToRoute( this -> getNode() );
				//
				// Ignore old RREQs, remember new ones. Transmit route reply.
				//
				if ( oldRREQ( m ) ) {
					return;
				}
				else {
					rememberRREQ( m );
				}
				m -> messageType = Message::RREP;
				transmit( m );
				return;
			}
			//
			// This node is just an intermediate node, thus forward the RREQ.
			//
			else { 
				//
				// Ignore own RREQ and previously forwarded RREQ
				//
				if ( ( this -> getNode() == m -> getSource() ) ||
					 ( includedInRoute(this -> getNode(), m -> getRoute()) ) ) {
					return;
				}
				//
				// RREQ originated elsewhere and was not seen before.
				// Add node to route and forward the message to all neighbors.
				//
				else {		
					m -> addToRoute( this -> getNode() );
					m -> sendNode = this -> getNode();
					++( m -> routeMarker );
					transmit( m );
				}
			}
		break;
		//
		// Route replies are handled according to the node's state.
		//
		case Message::RREP:
			//
			// If the node is idle, there can only be old RREPs, or it is
			// an intermediate node, in which case the message is sent
			// backwards alongs the route.
			//
			if ( !waiting ) {
				if ( this -> getNode() == m -> getSource() ) {
					return;
				}
				else {
					--( m -> routeMarker ); 
					transmit( m );
				}
			}
			//
			// If the node is waiting for a reply, the route table is 
			// updated and the list of expected RREPs is updated. The 
			// wait timer will be reset, if there are more RREPs still
			// expected to arrive. Otherwise, it will be stopped.
			//
			else if ( waiting ) {
		
				updateRouteTable( m );
				
				if ( this -> getNode() == m -> getSource() ) {
					
					if ( isExpectedRREP( m ) ) {
						expectedRREPs.erase( m -> sequenceNumber );
					}
					
					if ( expectedRREPs.size() == 0 ) { 
						RREQTimer -> stop();
					}
					else {
						RREQTimer -> reset( 40 );
					}
					//
					// Set the next hop and transmit DATA packet.
					//
					m -> messageType = Message::DATA;
					m -> routeMarker = 0;
					++( m -> routeMarker );
					transmit( m );
				}
				//
				// RREPs not for this node will simply be forwarded
				// backwards on the contained route.
				//
				else {
					--( m -> routeMarker ); 
					transmit( m );
				}
			}
		break;

		default:
			error( "SoftwareEntity::handleLowerLayerInput(): cannot determine message type" );
		}
	}
	//
	// Update the map routeTable, which contains destination 
	// nodes as keys and route pointers as values.
	//
	void updateRouteTable( Message* m ) {
		Route::iterator it = m -> getRoute() -> end();
		NetNode* dest = (*(--it));
		this -> routeTable[ dest ] = m -> getRoute();
	}
	//
	// Try to find an entry for the destination node in the route table,
	// returns 0 or whatever route is stored.
	//
	Route* lookupRoute( NetNode* dest ) {

		if ( routeTable.empty() )
			return 0;

		RouteTable::iterator iter;
		iter = routeTable.find( dest );
		
		return ( iter == routeTable.end() ) ? 
				 0 : (*iter).second; 
	}
	//
	// Check whether the route already contains this node.
	//
	bool includedInRoute( NetNode* searchNode, Route* route ) {
		Route::iterator iter;
		for ( iter = route -> begin(); iter != route -> end(); ++iter ) {
			if ( (*iter) == searchNode ) break;
		}
		return iter != route -> end();
	}
	//
	// Store RREQ in a list to mark it as previously seen request. 
	//
	void rememberRREQ( Message* m ) {
		seenRREQs.insert( m -> sequenceNumber );
	}
	//
	// Check whether this route request was seen already.
	//
	bool oldRREQ( Message* m ) {
		set< int >::iterator result = seenRREQs.find( m -> sequenceNumber );
		return result != seenRREQs.end();
	}
	//
	// Check whether the route reply is expected by this node.
	// This will effectively ignore old ones.
	//
	bool isExpectedRREP( Message* m ) {
		set< int >::iterator result = expectedRREPs.find( m -> sequenceNumber );
		return result != expectedRREPs.end();
	}
	//
	// Set the message type to RREQ, add the sequence number of this 
	// message to the list of the expected replies and remember the RREQ
	// to ignore the node's own requests. Then update the route with the 
	// first node and broadcast the message.
	//
	void sendRREQ( Message* m ) {
				
		if ( !RREQTimer -> isSet() ) 
			RREQTimer -> setIn( 40 );
		
		m -> messageType = Message::RREQ;
		expectedRREPs.insert( m -> sequenceNumber );
		rememberRREQ( m );
		
		// add this node to route and transmit
		m -> route = new Route;
		m -> route -> push_back( this -> getNode() );
		++( m -> routeMarker );
		transmit( m );
	}
	//
	// This function handles the waiting state for route replies.
	//	
	void waitForRREP( Message* m ) {
		Memo* theAlerter = 0;
		
		while( true ) {
			theAlerter = wait( RREQTimer, bottomPortHead(), topPortHead() );
			//
			// If a timeout occurs and the resend limit is not reached, send
			// the request again. Otherwise do nothing.
			//
			if ( theAlerter == RREQTimer ) {
			
				if ( m -> repeatedSends >= maxRepeatRREQ ) {
					return;
				}
				++( m -> repeatedSends );
				sendRREQ( m );
			}
			//
			// Handle incoming messages as usual and return to 
			// idle state when no more RREQs are expected to arrive.
			//
			else if ( theAlerter == bottomPortHead() ) {
				handleLowerLayerInput( readBottom() );
				if ( expectedRREPs.size() == 0 ) {
					return;
				}
			}
			//
			// Handle user data input as usual.
			//
			else if ( theAlerter == topPortHead() ) {
				
				handleUpperLayerInput( readTop() );
			}			
			else if ( theAlerter == 0 && isInterrupted() ) {
				warning( "SoftwareEntity::waitForRREP(): interrupted by another process" );
			}
			else
				error( "SoftwareEntity::waitForRREP(): unknown alerter" );
			
			theAlerter = 0;
		}
	}
};

//
// Initialize static members
//
int SoftwareEntity::messageCount = 0;
const int SoftwareEntity::maxRepeatRREQ = 2;
const int HardwareEntity::maxRepeatUCast = 2;


#endif /*PROTOCOLEXAMPLEENTITIES_H_*/
