//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProcessQueue.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/25

	\brief Implementation of ProcessQueue, ProcessOrder, 
	DefaultOrder and PriorityOrder

	\sa ProcessQueue.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/InSort.h>
#include <odemx/base/Process.h>
#include <odemx/base/ProcessQueue.h>

using namespace odemx;
using namespace std;

ProcessQueue::ProcessQueue(ProcessOrder* pred/* = &defOrder*/) : order(pred) {
}

ProcessQueue::~ProcessQueue() { }

Process* ProcessQueue::getTop() const {
	if (isEmpty())
		return 0;

	return *l.begin();
}

const std::list<Process*>& ProcessQueue::getList() const {
	return l;
}

bool ProcessQueue::isEmpty() const {
	return l.empty();
}

void ProcessQueue::popQueue() {
	remove(*l.begin());
}

void ProcessQueue::remove(Process* p) {
	assert(p!=0);

	// Process p forgets this queue
	p->dequeue(this);

	l.remove(p);
}

void ProcessQueue::inSort(Process* p,  bool fifo/* = true*/) {
	assert(p!=0);

	// Process p has to remember this queue
	p->enqueue(this);

	InSort<ProcessOrder&>(l, p, *order, fifo);
	
}

bool DefaultOrder::operator ()(const Process& f, const Process& s) {
	return f<s;
}

bool PriorityOrder::operator ()(const Process& f, const Process& s) {
	return (f.getPriority()>s.getPriority());
}

void odemx::awakeAll(ProcessQueue* q) {
	if (q->isEmpty())
			return;

	Process* currentProcess=q->getTop()->getCurrentProcess();

	std::list<Process*>::const_iterator i;
	for (i=q->getList().begin(); i!=q->getList().end(); ++i) {
		assert(*i!=currentProcess);

		(*i)->activateAfter(currentProcess);
	}
}

void odemx::awakeFirst(ProcessQueue* q) {
	if (q->isEmpty())
		return;

	Process* currentProcess=q->getTop()->getCurrentProcess();

	if (q->getTop()!=currentProcess)
		q->getTop()->activateAfter(currentProcess);
}

void odemx::awakeNext(ProcessQueue* q, Process* p) {
	if (q->isEmpty())
		return;

	Process* currentProcess=p->getCurrentProcess();

	std::list<Process*>::const_iterator i;
	for (i=q->getList().begin();
		i!=q->getList().end() && (*i)!=p;
		++i);

	if ((*i)==p) {
		++i;

		if (i!=q->getList().end() && (*i)!=currentProcess)
			(*i)->activateAfter(currentProcess);
	}
}

DefaultOrder odemx::defOrder;
PriorityOrder odemx::priOrder;
