//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Bin.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/26

	\brief Declaration of odemx::Bin

	\sa Bin.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_BIN_INCLUDED
#define ODEMX_BIN_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/Report.h>
#include <odemx/util/Observable.h>
#include <odemx/util/StatisticObject.h>
#include <odemx/statistics/Statistics.h>
#include <odemx/synchronization/Queue.h>

namespace odemx{
	class BinObserver;

	/** \class Bin

		\ingroup synch

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Bin is a single bounded (number of token greater or
		equal to zero) token abstract resource.

		\note Bin supports Observation
		\note Bin supports Trace
		\note Bin supports Report

		<!-- [\sa {references to other classes}]* -->

		Bin is a low bounded (token number >= 0) token abstract resource
		for synchronising Process objects. A process is blocked in take()
		if the requested number of token is greater than the available
		number of token. It is reactivated when enough token are given (back)
		to the resource. If multiple processes are waiting for reactivation
		process-priority and FIFO-strategy are used to choose the process.
		Bin is generally used for producer and consumer synchronisation.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Bin : public Observable<BinObserver>,
				public DefLabeledObject,
				public StatisticObject,
				public virtual TraceProducer,
				public virtual ReportProducer
	{
	public:
		/**
			\brief Construction
			\param s
				pointer to Simulation object
			\param l
				label for this object
			\param startTokenNumber
				initial number of token in Bin
			\param o
				initial observer
		*/
		Bin(Simulation* s, Label l,	unsigned int startTokenNumber,
			BinObserver* o = 0);
		/// Destruction
		~Bin();

		/**
			\name Token management

			@{
		*/
		/**
			\brief Take \p n token

			Takes \p n token from resource if possible. Otherwise
			the current process is blocked until enough token are
			available. If a blocked process is interrupted it is
			reactivated and the function returns 0. Without interruption
			take returns \p n.
		*/
		unsigned int take(unsigned int n);

		/**
			\brief Give \p n token

			Gives \p n token to resource. Blocked process objects
			could be activated by this call.
		*/
		void give(unsigned int n);

		/// Number of tokens available
		unsigned getTokenNumber() const {return tokenNumber;}
		//@}

		/// Get list of blocked processes
		const std::list<Process*>& getWaitingProcessList() const {return takeWait.getList();}

		/**
			\name Statistics

			@{
		*/
		virtual void reset();

		/// Number of takes
		unsigned int getUsers() const {return users;}

		/// Number of gives
		unsigned int getProviders() const {return providers;}

		/// Initial number of token
		unsigned int getInitial() const {return initToken;}

		/// Min number of token
		unsigned int getMin() const {return (unsigned int)tokenStatistics.getMin();}

		/// Max number of token
		unsigned int getMax() const {return (unsigned int)tokenStatistics.getMax();}

		/// Average free token
		double getAVFreeToken() const {return tokenStatistics.getMean();}

		/// Average waiting time
		double getAVWaitTime() const;

		//@}

		/// Get pointer to trace
		virtual Trace* getTrace() const {return env;}

		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace Bin events.
			A TraceConsumer can use these constants to identify trace
			events send	by Bin.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markTakeFail;
		static const MarkType markTakeSucceed;
		static const MarkType markGive;

		static const MarkType markChangeTokenNumber;

		static const TagId baseTagId;

		static const Tag tagCurrent;
		static const Tag tagTokenNumber;
		//@}

		/// Generate report
		virtual void report(Report* r);

		// Implementation
	private:
		/// simulation
		Simulation* env;

		/// available number of token
		unsigned int tokenNumber;

		// statistics
		/// initialise token number
		unsigned int initToken;

		/// token statistics
		Accum tokenStatistics;

		/// user statistic
		unsigned int users;
		unsigned int providers;
		double sumWaitTime;

		/// process management
		Queue takeWait;

		// help methods
		/// get current process
		Process* getCurrentProcess() {return env->getCurrentProcess();}
		/// trace marks
		void traceTSFG(MarkType m, unsigned int n);
	};

	/** \interface BinObserver

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for Bin specific events.

		<!-- [\note {notes}]* -->

		\sa Bin

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class BinObserver {
	public:
		virtual ~BinObserver() { }
		/// Creation
		virtual void onCreate(Bin* sender) {}
		/// Destruction
		virtual void onDestroy(Bin* sender) {}

		/// Failed attempt to take n token
		virtual void onTakeFail(Bin* sender, unsigned int n) {}
		/// Successful attempt to take n token
		virtual void onTakeSucceed(Bin* sender, unsigned int n) {}
		/// Return of n token
		virtual void onGive(Bin* sender, unsigned int n) {}

		/// Change of token number
		virtual void onChangeTokenNumber(Bin* sender, unsigned int oldTokenNumber,
			unsigned int newTokenNumber) {}
	};
}

#endif

