//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Bin.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/26

	\brief Implementation of classes in Bin.h

	\sa Bin.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/InSort.h>
#include <odemx/util/ErrorHandling.h>
#include <odemx/synchronization/Bin.h>

#include <string>
#include <cassert>
#include <algorithm>

using namespace std;
using namespace odemx;

Bin::Bin(Simulation* s, Label l, unsigned int startTokenNumber, BinObserver* o/* = 0*/) :
	Observable<BinObserver>(o), StatisticObject(s),
	env(s), tokenNumber(startTokenNumber), initToken(startTokenNumber),
	tokenStatistics(s, (string(l) + "_tokenStatistic").c_str()),
	takeWait(s, (string(l) + string("_queue")).c_str(), &priOrder)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	// observer
	_obsForEach(BinObserver, Create(this));

	// Init Statistics
	reset();
}

Bin::~Bin() {
	// trace
	getTrace()->mark(this, markDestroy);

	// observer
	_obsForEach(BinObserver, Destroy(this));
}

// Token
unsigned int Bin::take(unsigned int n) {
	// compute order of service
	takeWait.inSort(getCurrentProcess());

	if (n>tokenNumber ||
		takeWait.getTop()!=getCurrentProcess()) {
		// not enough token or not the
		// first process to serve

		// trace
		traceTSFG(markTakeFail, n);

		// observer
		_obsForEach(BinObserver, TakeFail(this, n));

		// statistics
		SimTime waitTime=env->getTime();

		// block execution
		while (n>tokenNumber || takeWait.getTop()!=getCurrentProcess()) {
			getCurrentProcess()->sleep();

			if (getCurrentProcess()->isInterrupted()) {
				takeWait.remove(getCurrentProcess());
				return 0;
			}
		}

		// block released
		// statistics
		sumWaitTime+=env->getTime()-waitTime;
	}

	// trace
	getTrace()->mark(this, markChangeTokenNumber, tokenNumber-n, tokenNumber);

	// observer
	_obsAForEach(BinObserver, TokenNumber, tokenNumber, tokenNumber-n);

	// take token
	tokenNumber -= n;

	// statistics
	users++;
	tokenStatistics.update(tokenNumber);

	// trace
	traceTSFG(markTakeSucceed, n);

	// observer
	_obsForEach(BinObserver, TakeSucceed(this, n));

	// remove from list
	takeWait.remove(getCurrentProcess());

	// awake next process
	awakeFirst(&takeWait);

	return n;
}

void Bin::give(unsigned int n) {
	// trace
	getTrace()->mark(this, markChangeTokenNumber, tokenNumber+n, tokenNumber);

	// observer
	_obsAForEach(BinObserver, TokenNumber, tokenNumber, tokenNumber+n);

	// releas token
	tokenNumber += n;

	// statistics
	providers++;
	tokenStatistics.update(tokenNumber);

	// trace
	traceTSFG(markGive, n);

	// observer
	_obsForEach(BinObserver, Give(this, n));

	// awake process waiting for give
	awakeFirst(&takeWait);
}

void Bin::reset() {
	// reset time
	StatisticObject::reset();

	// reset queue statistics
	takeWait.reset();

	// Accum reset and re-init
	tokenStatistics.reset();
	tokenStatistics.update(getTokenNumber());

	users=providers=0;
	sumWaitTime=0;
}

double Bin::getAVWaitTime() const {
	return sumWaitTime/getUsers();
}

void Bin::report(Report* r) {
	assert(r);
	takeWait.report(r);

	static const char* labels[] = {"Name", "Reset at", "Queue", "Users", "Provider", "Init number of token", "Min  number of token", "Max number of token", "Now number of token",
								   "Avg number of token", "Avg waiting time"};
	static const ColumnType types[] = {STRING, REAL, STRING, INTEGER, INTEGER, INTEGER, INTEGER, INTEGER, INTEGER,
									   REAL, REAL};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Bin Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << takeWait.getLabel() << getUsers() << getProviders();
	*t << getInitial() << getMin() << getMax() << getTokenNumber();
	*t << getAVFreeToken() << getAVWaitTime();
}

void Bin::traceTSFG(MarkType m, unsigned int n) {
	getTrace()->beginMark(this, m);
	getTrace()->addTag(tagCurrent, getCurrentProcess());
	getTrace()->addTag(tagTokenNumber, n);
	getTrace()->endMark();
}

const MarkTypeId Bin::baseMarkId = 1000;

const MarkType Bin::markCreate = MarkType("create", baseMarkId+1, typeid(Bin));
const MarkType Bin::markDestroy = MarkType("destroy", baseMarkId+2, typeid(Bin));

const MarkType Bin::markTakeFail = MarkType("takeFail", baseMarkId+10, typeid(Bin));
const MarkType Bin::markTakeSucceed = MarkType("takeSucceed", baseMarkId+11, typeid(Bin));
const MarkType Bin::markGive = MarkType("give", baseMarkId+12, typeid(Bin));

const MarkType Bin::markChangeTokenNumber = MarkType("changeTokenNumber", baseMarkId+20, typeid(Bin));

const TagId Bin::baseTagId = 1000;

const Tag Bin::tagCurrent = Tag(baseTagId+1);
const Tag Bin::tagTokenNumber = Tag(baseTagId+2);

