//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file SimpleProtocol.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/06/10

	\brief Example for ODEMx protocol simulation techniques

	\sa SimpleProtocolMessage.h SimpleProtocolEntities.h

	This example shows essential techniques for  
	implementing a protocol simulation with ODEMx.
	Protocol behavior is specified by employing the OSI
	layer model.

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/

/** 
	\example SimpleProtocol.cpp

	With this example, the essential protocol simulation techniques of 
	ODEMx are introduced.

	Protocol models for ODEMx are based on the OSI layer model approach,
	where the protocol stack is separated into several layers, each with 
	a distinct function.

	A protocol simulation starts out with a ProtocolSimulation object, 
	which is basically a normal Simulation object, extended by functions
	to create and access the network topology, the transmission medium,
	and the protocol layers. 
	
	The network topology is described by NetNode objects and their
	connections, which are maintained by the class NetTopology. Derived 
	from this class is the TransmissionMedium, which also handles
	message transmissions between the network's nodes.
	
	A node itself has an input and an output port, which can be used 
	by the system environment to pass data to a node, and to register 
	successful message transmissions at a receiving node. The behavior
	of a node is modeled in its active components, so-called entities.
	
	A ProtocolEntity is a process representing a protocol layer function.
	Each node maintains exactly one entity per layer, which are 
	automatically created upon NetNode construction. Entities can receive
	input from the layer above or below and a behavior must be specified
	according to the data passed between these layers.

	In this example, a simple protocol simulation is defined, which shows
	the essential techniques to model and run a protocol simulation with
	ODEMx. Here, two protocol layers are modeled, one for the hardware
	part of a node, and one for the software part. The network contains 
	simple nodes, whose behavior is modeled by the definition of a
	HardwareEntity and a SoftwareEntity. The network's topology is
	definied in the simulation objects's initialization function. Lastly,
	messages between nodes are triggered by external events, which are also
	scheduled during the initialization phase.
	
	\sa SimpleProtocolMessage.h SimpleProtocolEntities.h
*/


#include <odemx/odemx.h>
#include "SimpleProtocolMessage.h"
#include "SimpleProtocolEntities.h"

#include <iostream>
#include <string>
#include <vector>


using namespace odemx;
using namespace std;


//
// In this example, a simple protocol simulation is defined, which shows
// the essential techniques to model and run a protocol simulation with
// ODEMx. There are two protocol layers, one for the hardware
// part of a node, and one for the software part. The network contains 
// simple nodes, whose behavior is modeled by defining a
// HardwareEntity and a SoftwareEntity. The network's topology is
// definied in the simulation objects's initialization function. Lastly,
// messages between nodes are triggered by external events, which are also
// scheduled during the initialization phase.
//
// NetNodes are basic components of protocol simulations. In this case, 
// a very simple derivation is used.
//
class SimpleNetNode: public NetNode {
public:
	SimpleNetNode( ProtocolSimulation* sim, Label l ):
		NetNode( sim, l ) {}
	
protected:
	//
	// Defines the life cycle of a node. In this case, a node simply waits
	// for input from the environment, upon which it is awakened and
	// transmits the given data.
	//
	virtual int main() {
		while( true ) {
			//
			// A node will wait until an element is put into its
			// input port, which will wake the node.
			//
			Memo* theAlerter = wait( inputPortHead() );   
			UserData* data;
			//
			// Check the alerter. If it is unknown, do nothing 
			// and simply wait again.
			//
			if ( theAlerter == inputPortHead() ) {
				data = (UserData*)readInputPort();
			}
			else {
				error( "SimpleNetNode::main(): unknown alerter" );
				continue;
			}
			//
			// A node's transmit function hands the given data to its
			// top layer entity, which will then handle this input.
			//
			transmit( data );
		}
		return 0;
	}
};


//
// For demonstration purposes, a process modeling the environment
// is defined. This class 
//
class EnvironmentReceiver: public Process {
public:
	//
	// Upon construction, register all nodes' output Ports (Memo derivations)
	//
	EnvironmentReceiver( ProtocolSimulation* sim, Label label ):
		Process( sim, label )
		{
			NodeVector& nodes = sim -> getNetTopology() -> getAllNodes();
			NodeVector::iterator iter;
			for( iter = nodes.begin(); iter != nodes.end() ; ++iter ) {
				portVector.push_back( ( *iter ) -> outputPort() );
			}
		}
protected:
	//
	// Life cycle of this receiver process. It waits until an alert
	// notifies this process of the reception of a message, which is
	// then printed to stdout.
	//
	virtual int main() {
		while( true ) {
		
			Memo* theAlerter = wait( &portVector );
			
			MemoVector::iterator iter;
			for( iter = portVector.begin(); iter != portVector.end() ; ++iter ) {
				
				PortHead* currentPort = (PortHead*)(*iter);

				if ( theAlerter == currentPort ) {
					UserData* msgData = (UserData*) currentPort -> get();
					
					cout <<  "Message received: \"" << msgData -> content
						 << "\" by " << currentPort -> getLabel() 
						 << " at SimTime " << getCurrentTime() << endl;
				}
			}
		}
		return 0;
	}
private:
	MemoVector portVector;
};


//
// Triggering environment data input and message transmissions 
// can be done using events. 
//
class MessageEvent: public Event {
public:
	MessageEvent( ProtocolSimulation* sim, 
				  NetNode* src, NetNode* dest, string msg ): 
		Event( sim, "Message Event" ),
		source( src ),
		destination( dest ),
		message( msg )
		{}
protected:
	//
	// The event action is executed when the simulation reaches
	// its given excution time. At that point, a simple message
	// is handed to the source node, which will initiate a transmission
	// of the message to the destination node.
	//
	virtual void eventAction() {

		cout << "Message event: \"" << message
			 << "\" from " << source -> getLabel()
			 << " to " << destination -> getLabel()
			 << " at SimTime " << getCurrentTime() << endl;
		
		UserData* udat = new UserData( source, destination, message );
		source -> input( udat );
	}
private:
	NetNode* source;
	NetNode* destination;
	string message;
};


//
// The simulation object initalizes all protocol components
// and schedules message events.
//
class SimpleProtocolSimulation: public ProtocolSimulation {
public:
	SimpleProtocolSimulation( Label l):
		ProtocolSimulation( l )
		{}

	virtual ~SimpleProtocolSimulation()
		{
			vector< Event* >::iterator iter;
			for( iter = messages.begin(); iter != messages.end(); ++iter )
				delete *iter;
		}
	//
	// This function creates the required simulation components
	// such as nodes, layers, and message events. The network's
	// topology is defined here. The topology could easily be 
	// changed during simulation by creating special events for 
	// this action, or by having another process influence the 
	// network status.
	//
	virtual void initSimulation() {
		
		cout << "Initializing necessary processes..." << endl;
		//
		// Every protocol simulation requires a transmission
		// medium. It is, however, not necessary to derive one's
		// own class from the base TransmissionMedium. An object 
		// of this class can be used as default transmission medium.
		// 
		TransmissionMedium* medium = new TransmissionMedium( this, "Medium" );
		//
		// Layers are created by a template function that requires
		// the corresponding entity type as parameter. All protocol
		// entities will be initialized automatically upon NetNode
		// construction, which is why all layers must be defined
		// prior to creating any nodes.
		//
		// Furthermore, the layers must be in top-down order because the
		// entities of a node are linked automatically to communicate 
		// with the layers above and below.
		//
		createLayer< SoftwareEntity >( "Software" );
		createLayer< HardwareEntity >( "Hardware" );
		//
		// Creation of all nodes. For simplicity, this has also
		// been wrapped by a function, as nodes must be created,
		// registered with the NetTopology, and scheduled for 
		// initialization. The given string is the node's label.
		//
		NetNode* A = initNode( "A" );
		NetNode* B = initNode( "B" );
		NetNode* C = initNode( "C" );
		NetNode* D = initNode( "D" );
		NetNode* E = initNode( "E" );
		NetNode* F = initNode( "F" );
		//
		// Next, the topology is built by connecting the newly 
		// created nodes. Connections are set by giving a probability
		// for message delivery, which will be modeled internally by
		// a random variable. Both directions can be given separately,
		// as setConnection is overloaded with a fourth double parameter. 
		//
		medium -> setConnection( A, B, 0.9 );
		medium -> setConnection( B, C, 0.9 );
		medium -> setConnection( A, C, 1.0 );
		medium -> setConnection( C, D, 1.0 );
		medium -> setConnection( D, E, 0.8 );
		medium -> setConnection( D, F, 0.9 );
		medium -> setConnection( E, F, 0.9 );
		//
		// Lastly, the receiving environment process is created, which
		// uses the node information maintained by NetTopology.
		//
		Process* receiver = new EnvironmentReceiver( this, "ENV-Receiver" );
		receiver -> hold();
		//
		// Show the connectivy matrix with connection quality.
		//
		cout << "----------------------------" << endl
			 << "Current connectivity matrix:" << endl << endl;
		medium -> printNodeMatrix();
		//
		// Schedule message events to occur during simulation.
		// This is also done by a wrapper function, which stores
		// messages in a vector for memory management.
		//
		scheduleMessageEvent( A, D, "message from A to D", 10 );
		scheduleMessageEvent( A, F, "message from A to F", 15 );
		scheduleMessageEvent( B, F, "message from B to F", 13 );
		scheduleMessageEvent( E, A, "message from E to A", 9 );
		
		printExecutionList();
		cout << "Initialization finished." << endl;
	}
	
	void scheduleMessageEvent( NetNode* src, NetNode* dest, string msg, SimTime t ) {
		messages.push_back( new MessageEvent( this, src, dest, msg ) );
		messages.back() -> scheduleAt( t );
	}
	
	NetNode* initNode( Label node ) {
		NetNode* newNode = new SimpleNetNode( this, node );
		getNetTopology() -> registerNode( newNode );
		newNode -> hold();
		return newNode;
	}
	
private:
	vector< Event* > messages;
};

//
// As almost everything is specified in the simulation, the main
// function only handles ODEMx monitoring objects such as the
// HtmlTrace, XmlTrace, and a simulation report.
//
int main( int argc, char* argv[] ) {
 	
	ProtocolSimulation* sim = new SimpleProtocolSimulation( "Example Protocol Simulation" );
	
	HtmlTrace* htmlTrace = new HtmlTrace( sim, "ProtocolHtmlTrace.html" );
	
	XmlTrace* xmlTrace = new XmlTrace( sim, "ProtocolXmlTrace", 1000 );

	HtmlReport* reporter = new HtmlReport( sim, "ProtocolHtmlReport.html" );
	reporter -> addProducer( sim );
	
	sim -> addConsumer( htmlTrace );
	sim -> addConsumer( xmlTrace );
	sim -> startTrace();
	
	cout << endl << "Starting " << sim -> getLabel() << endl << endl;
	cout << "============================================" << endl;
	
	sim -> run();

	cout << "============================================" << endl;
	cout << "Finished " << sim -> getLabel() << endl << endl;
	
	reporter -> generateReport();
	sim -> stopTrace();
	
	delete htmlTrace;
	delete xmlTrace;
	delete reporter;
}
