//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ExecutionList.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/25

	\brief Implementation of ExecutionList

	\sa ExecutionList.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/Event.h>
#include <odemx/base/Process.h>
#include <odemx/base/ExecutionList.h>
#include <odemx/base/InSort.h>
#include <odemx/util/ErrorHandling.h>

#include <iostream>
#include <algorithm>

using namespace std;
using namespace odemx;

ExecutionList::ExecutionList(ExecutionListObserver* o) : Observable<ExecutionListObserver>(o) {
	_obsForEach(ExecutionListObserver, Create(this));
}

ExecutionList::~ExecutionList() {
	_obsForEach(ExecutionListObserver, Destroy(this));
}

void ExecutionList::addSched(Sched* p) {
	// attach Sched p at SimTime to ExecutionList (FIFO), considering priority

	if (p==0) {
		// error: illegal parameter
		error("addSched(); invalid argument");
		return;
	}

	// adjust execution time
	if (p->getExecutionTime() < getTime())
		p->setExecutionTime(getTime());

	inSort(p);

	_obsForEach(ExecutionListObserver, AddSched(this, p));
}

void ExecutionList::insertSched(Sched* p) {
	// insert Sched p at SimTime in ExecutionList (LIFO), considering priority

	if (p==0) {
		// error: illegal parameter
		error("insertSched(); invalid argument");
		return;
	}

	// adjust execution time
	if (p->getExecutionTime() < getTime())
		p->setExecutionTime(getTime());

	inSort(p, false);
	_obsForEach(ExecutionListObserver, InsertSched(this, p));	
}

void ExecutionList::insertSchedAfter(Sched* p, Sched* previous) {
	// insert Sched p after 'previous', changing priority if necessary

	std::list<Sched*>::iterator pre;
	std::list<Sched*>::iterator i;
	
	if (p==0) {
		// error: no Sched to enter
		error("insertSchedAfter(); invalid argument: Sched* p");
		return;
	}

	if (previous==0) {
		// error: no 'previous'
		error("insertSchedAfter(); invalid argument: Sched* previous");
		return;
	}

	if (p==previous)
		return;

	// find 'previous'
	pre = find(l.begin(), l.end(), previous);
	if (pre==l.end()) {
		// error: 'previous' is not listed
		error("insertSchedAfter(); 'previous' not listed");
		return;
	}

	// remove Sched p from ExecutionList
	i = find(l.begin(), l.end(), p);
	if (i!=l.end())
		l.erase(i);

	// adjust priority
	if (p->getPriority()>(*pre)->getPriority())
		p->setPriority((*pre)->getPriority());

	// set execution time
	p->setExecutionTime((*pre)->getExecutionTime());

	// insert Sched p after 'pre' (before 'next' of 'pre')
	++pre;
	l.insert(pre, p);

	_obsForEach(ExecutionListObserver, InsertSchedAfter(this, p, previous));
}

void ExecutionList::insertSchedBefore(Sched* p, Sched* next) {
	// insert Sched p before 'next', changing priority if necessary

	std::list<Sched*>::iterator nex;
	std::list<Sched*>::iterator i;
	
	if (p==0) {
		// error: no Sched to enter
		error("insertSchedBefore(); invalid argument: Sched* p");
		return;
	}

	if (next==0) {
		// error: no 'next'
		error("insertSchedBefore(); invalid argument: Sched* next");
		return;
	}

	if (p==next)
		return;

	// find 'next'
	nex = find(l.begin(), l.end(), next);
	if (nex==l.end()) {
		// error: 'next' is not listed
		error("insertSchedBefore(); 'next' not listed");
		return;
	}

	// remove Sched p from ExecutionList
	i = find(l.begin(), l.end(), p);
	if (i!=l.end())
		l.erase(i);

	// adjust priority
	if (p->getPriority()<(*nex)->getPriority())
		p->setPriority((*nex)->getPriority());

	// set execution time
	p->setExecutionTime((*nex)->getExecutionTime());

	// insert Sched p before 'next'
	l.insert(nex, p);

	_obsForEach(ExecutionListObserver, InsertSchedBefore(this, p, next));
}

void ExecutionList::removeSched(Sched* p) {
	l.remove(p);

	_obsForEach(ExecutionListObserver, RemoveSched(this, p));
}

Sched* ExecutionList::getNextSched() {
	if(isEmpty())
		return 0;
	
	return l.front();
}

void ExecutionList::inSort(Sched* e, bool fifo/* = true*/) {
	InSort(l, e, fifo);
}


void ExecutionList::printExecutionList() {
	list< Sched* >::iterator iter;

	cout << "--------------" << endl;
	cout << "ExecutionList:" << endl; 

	for ( iter = l.begin(); iter != l.end(); ++iter ) {

		if ( (*iter) -> getSchedType() == Sched::PROCESS ) {
			cout << "  Process \"" << (*iter) -> getLabel()  
				 << "\" scheduled at " << (*iter) -> getExecutionTime(); 
		}
		
		if ( (*iter) -> getSchedType() == Sched::EVENT ) {
			cout << "  Event \"" << (*iter) -> getLabel()  
				 << "\" scheduled at " << (*iter) -> getExecutionTime(); 
		}
		cout << endl;
	}
	cout << "--------------" << endl;
}
