#ifndef _SUFFIXDISKPAGEPOOL_HPP_
#define _SUFFIXDISKPAGEPOOL_HPP_

#include "globals.hpp"
#include <fstream>
#include "string.h"

/**
 * Implementiert den Dateipuffer für die Suffixe.
 */
class SuffixDiskPage {

	// PosID, bei der die Seite anfängt
	POS_TYPE pos_start;

	// Anzahl der Knoten in der Seite (muß durch
	POS_TYPE pos_end;

	// Zeiger auf die Rohdaten
	char * mPage;

	// Schreibezeiger; wird bei jeder Schreiboperation weitergesetzt
	char * put_ptr;
	POS_TYPE pos_put;

	// Lesezeiger; wird bei jeder Leseoperation weitergesetzt
	char * get_ptr;
	POS_TYPE pos_get;

	// flag, ob in die Seite geschrieben wurde
	bool dirty;

public:
	// C'tor, allokiert gleich den Speicher für die Seite
	SuffixDiskPage(POS_TYPE p_pos_start):
		pos_start(p_pos_start), pos_end(PTREE_NULLPOS), dirty(false){mPage = new char[nsSuffixPageSize];}

	//D'tor, gibt den Speicher für die Seite wieder frei
	~SuffixDiskPage(){delete mPage;}

	// true, wenn der Platz durch alle Knoten besetzt ist
	bool isFull(){return pos_end - pos_start > nsSuffixPageSize;}

	// true, auf die Seite geschrieben wurde
	bool isDirty(){return dirty;}

	// gibt die Startposition zurück
	POS_TYPE getStartPos(){return pos_start;}

	// setzt den Schreibezeiger
	void seekp(POS_TYPE pos){put_ptr = mPage + (pos - pos_start);pos_put = pos;}
	// setzt den Lesezeiger
	void seekg(POS_TYPE pos){get_ptr = mPage + (pos - pos_start);pos_get = pos;}

	// schreibt count chars von source in die Speicherseite (an die aktuelle Position des put_Pointers)
	// setzt put_ptr und pos_end neu
	void write(char * source, unsigned int count);

	// liest count chars von der Position des Get-Pointers, schreibt das Ergebnis nach target
	// es findet kein range-check statt!
	void read(char * target, unsigned int count){memcpy(target, get_ptr, count);get_ptr+=count;}

	// ganze Seite schreiben/lesen
#ifdef USE_C_FILES
	void writePage(FILE * & c_file);
	void readPage(FILE * & c_file);
#else
	void writePage(fstream& stream);
	void readPage(fstream& stream);
#endif
};

/**
 * Verwaltet die Speicherseiten
 */
class SuffixDiskPagePool {

	typedef map<POS_TYPE, SuffixDiskPage*> SuffixPagePoolBuffer;
	typedef map<POS_TYPE, SuffixDiskPage*>::iterator SuffixPagePoolBufferIterator;
	// Referenz auf den FileStream des Baumes
	// mal schauen ob das so funktioniert; ansonsten einen Zeiger verwenden
#ifdef USE_C_FILES
	FILE * & c_suffixFile;
#else
	fstream& suffixFile;
#endif

	// Map für <Anfang der Speicher, Zeiger auf DiskPage>
	// das ist quasi der Cache
	SuffixPagePoolBuffer pagePool;

	// LRU Verwaltung
	LRUList lruList;

	// aktuelle DiskPage, in die geschrieben oder von der gelesen wird
	SuffixDiskPage * currentPutPage;
	SuffixDiskPage * currentGetPage;

	// Schreibezeiger (Positionsbasiert)
	POS_TYPE pos_put;
	// Lesezeiger (Positionsbasiert)
	POS_TYPE pos_get;

	// aktuelles Größe des Files --> neue Position für neuen Knoten
	POS_TYPE append_pos;

public:
#ifdef USE_C_FILES
	SuffixDiskPagePool(FILE * & p_c_suffixfile): c_suffixFile(p_c_suffixfile){append_pos = getFileSize();}
#else
	SuffixDiskPagePool(fstream& p_suffixfile): suffixFile(p_suffixfile){append_pos = getFileSize();}
#endif
	~SuffixDiskPagePool();

	void seekp(POS_TYPE pos);
	void seekg(POS_TYPE pos);

	// Position für einen neuen Knoten
	// die Längenangabe soll verhindern, daß ein String über die Blockgrenze geschrieben bzw. gelesen wird
	// --> kein Suffix ist länger als 256k Zeichen, daher wird immer nur auf einem Block operiert
	// --> damit muß nur an dieser Stelle etwas geändert werden im Vergleich zur Knoten-Behandlung
	POS_TYPE getAppendPos(LENGTH_TYPE length){
		if((append_pos % nsSuffixPageSize) + length >= nsSuffixPageSize){
			// Sonderfall, am Anfang der neuen Seite anfangen
			return append_pos = (((POS_TYPE)append_pos / (POS_TYPE)nsSuffixPageSize) + (POS_TYPE)1) * (POS_TYPE)nsSuffixPageSize;
		}else{
			// Normal
			return append_pos;
		}
	}

	/**
	 * schreibt count chars von source in die Speicherseite (an die aktuelle Position des put_Pointers)<br/>
	 * put_pointer rutscht automatisch count chars weiter
	 */
	void write(char * source, unsigned int count){
		currentPutPage->write(source,count);
		pos_put+=count;
		if(pos_put>append_pos){append_pos=pos_put;}
	}

	/**
	 * liest count chars von der Position des Get-Pointers<br/>
	 * get_pointer rutscht automatisch count chars weiter
	 */
	void read(char * target, unsigned int count){
		currentGetPage->read(target,count);
		pos_put+=count;
	}

private:

	// holt, wenn vorhanden, den Knoten für die gegebene Startposition aus dem Pool
	// NULL, wenn nicht drin (Cache miss)
	SuffixDiskPage * getFromPool(POS_TYPE startPos);

	// fügt die Seite in den Pool ein
	// muß sich im nächsten Schritt auch im die Ersetzungsstrategie kümmern
	bool insertIntoPool(SuffixDiskPage * page);

	// getFileSize()
#ifdef USE_C_FILES
	POS_TYPE getFileSize(){fseek(c_suffixFile, 0, SEEK_END);return ftell(c_suffixFile);}
#else
	POS_TYPE getFileSize(){suffixFile.seekg(0, ios::end);return suffixFile.tellg();}
#endif

};

#endif /*_SUFFIXDISKPAGEPOOL_HPP_*/
