#ifndef _DISKPAGEPOOL_HPP_
#define _DISKPAGEPOOL_HPP_

#include "globals.hpp"
#include "string.h"////////////
#include <fstream>

/**
 * Puffer-Klassen für die Baum-Knoten.
 *
 * DiskPage repräsentiert eine Seite, DiskPagePool kontrolliert die Seiten im Speicher.
 *
 */

class DiskPage {

	// PosID, bei der die Seite anfängt
	POS_TYPE pos_start;

	// Anzahl der Knoten in der Seite (muß durch
	POS_TYPE pos_end;

	// Zeiger auf die Rohdaten
	char * mPage;

	// Schreibezeiger; wird bei jeder Schreiboperation weitergesetzt
	char * put_ptr;
	POS_TYPE pos_put;

	// Lesezeiger; wird bei jeder Leseoperation weitergesetzt
	char * get_ptr;
	POS_TYPE pos_get;

	// flag, ob in die Seite geschrieben wurde
	bool dirty;

public:
	// C'tor, allokiert gleich den Speicher für die Seite
	DiskPage(POS_TYPE p_pos_start):
		pos_start(p_pos_start), pos_end(PTREE_NULLPOS), dirty(false){mPage = new char[nsNodePageSize /*PAGESIZE_ON_DISK*/];}

	//D'tor, gibt den Speicher für die Seite wieder frei
	~DiskPage(){delete mPage;}

	// true, wenn der Platz durch alle Knoten besetzt ist
	bool isFull(){return pos_end - pos_start > nsNodePageSize;}

	// true, auf die Seite geschrieben wurde
	bool isDirty(){return dirty;}

	// gibt die Startposition zurück
	POS_TYPE getStartPos(){return pos_start;}

	// setzt den Schreibezeiger
	void seekp(POS_TYPE pos){put_ptr = mPage + (pos - pos_start);pos_put = pos;}
	// setzt den Lesezeiger
	void seekg(POS_TYPE pos){get_ptr = mPage + (pos - pos_start);pos_get = pos;}

	// schreibt count chars von source in die Speicherseite (an die aktuelle Position des put_Pointers)
	// setzt put_ptr und pos_end neu
	void write(char * source, unsigned int count);

	// liest count chars von der Position des Get-Pointers, schreibt das Ergebnis nach target
	// es findet kein range-check statt!
	void read(char * target, unsigned int count){
		//cerr << "DiskPage::read(" << (int)get_ptr << ", " << count << ")" << endl;
		memcpy(target, get_ptr, count);get_ptr+=count;}

	// ganze Seite schreiben/lesen
#ifdef USE_C_FILES
	void writePage(FILE * & c_file);
	void readPage(FILE * & c_file);
#else
	void writePage(fstream& stream);
	void readPage(fstream& stream);
#endif
};

/**
 * Verwaltet die Speicherseiten
 */
class DiskPagePool {

	typedef map<POS_TYPE, DiskPage*> PagePoolBuffer;
	typedef map<POS_TYPE, DiskPage*>::iterator PagePoolBufferIterator;
	// Referenz auf den FileStream des Baumes
	// mal schauen ob das so funktioniert; ansonsten einen Zeiger verwenden
#ifdef USE_C_FILES
	FILE * & c_baumFile; // Referenz auf den Pointer
#else
	fstream& baumFile;
#endif

	// Map für <Anfang der Speicher, Zeiger auf DiskPage>
	// das ist quasi der Cache
	PagePoolBuffer pagePool;

	// LRU Verwaltung
	LRUList lruList;

	// aktuelle DiskPage, in die geschrieben oder von der gelesen wird
	DiskPage * currentPutPage;
	DiskPage * currentGetPage;

	// Schreibezeiger (Positionsbasiert)
	POS_TYPE pos_put;
	// Lesezeiger (Positionsbasiert)
	POS_TYPE pos_get;

	// aktuelles Größe des Files --> neue Position für neuen Knoten
	POS_TYPE append_pos;

public:
#ifdef USE_C_FILES
	DiskPagePool(FILE * & p_c_baumfile): c_baumFile(p_c_baumfile){append_pos = getFileSize();}
#else
	DiskPagePool(fstream& p_baumfile): baumFile(p_baumfile){append_pos = getFileSize();}
#endif
	~DiskPagePool();

	// setzt schreib/lesezeiger
	// ... und bereitet das Lesen/Schreiben vor
	void seekp(POS_TYPE pos);
	void seekg(POS_TYPE pos);

	// Position für einen neuen Knoten
	//POS_TYPE getAppendPos(){return append_pos;}
	POS_TYPE getAppendPos(){
		// egal wie groß der zu schreibende Knoetn wirklich ist: wenn der nicht mehr in den fixed node passen
		// würde wird eine neue Seite angelegt
		if((append_pos % nsNodePageSize) + nsNodeSizeOnDisk >= nsNodePageSize){
			// Sonderfall, am Anfang der neuen Seite anfangen
			return append_pos = (((POS_TYPE)append_pos / (POS_TYPE)nsNodePageSize) + (POS_TYPE)1) * (POS_TYPE)nsNodePageSize;
		}else{
			// Normal
			return append_pos;
		}
	}


	/**
	 * schreibt count chars von what in die Speicherseite (an die aktuelle Position des put_Pointers)<br/>
	 * put_pointer rutscht automatisch count chars weiter
	 */
	void write(char * source, unsigned int count){currentPutPage->write(source,count);pos_put+=count;if(pos_put>append_pos){append_pos=pos_put;}}

	/**
	 * liest count chars von der Position des Get-Pointers<br/>
	 * get_pointer rutscht automatisch count chars weiter
	 */
	void read(char * target, unsigned int count){currentGetPage->read(target,count);pos_get+=count;} // von pos_put nach pos_get geändert

private:

	/**
	 * holt, wenn vorhanden, den Knoten für die gegebene Startposition aus dem Pool
	 * NULL, wenn nicht drin (Cache miss)
	 */
	DiskPage * getFromPool(POS_TYPE startPos);

	/**
	 * fügt die Seite in den Pool ein
	 * muß sich im nächsten Schritt auch im die Ersetzungsstrategie kümmern
	 */
	bool insertIntoPool(DiskPage * page);

	// getFileSize()
#ifdef USE_C_FILES
	POS_TYPE getFileSize(){fseek(c_baumFile, 0, SEEK_END);return ftell(c_baumFile);}
#else
	POS_TYPE getFileSize(){baumFile.seekg(0, ios::end);return baumFile.tellg();}
#endif

};

#endif /* _DISKPAGEPOOL_HPP_*/

